#include "traffic_light.hpp"
/**
 * @file traffic_light.cpp
 * @author zhoutong (zhoutotong@qq.com)
 * @brief
 * @version 0.1
 * @date 2024-08-21
 *
 * @copyright Copyright (c) 2024
 *
 */
#include "api/traffic_light.hpp"
#include <chrono>

static uint64_t get_timestamp()
{
    return std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count();
}

TrafficLight::TrafficLight(const std::string &id, const double lat, const double lon, const double alt)
  : id_(id)
  , lat_(lat)
  , lon_(lon)
  , alt_(alt)
  , cb_(nullptr)
  , pub_(nullptr)
  , sub_(nullptr)
{
}

TrafficLight::TrafficLight(const std::string &id, const ControlCallback& cb, const double lat, const double lon, const double alt)
  : id_(id)
  , lat_(lat)
  , lon_(lon)
  , alt_(alt)
  , cb_(cb)
  , pub_(std::make_shared<mqtt_dataio::publisher<fleet::RoadsideInfo>>("/traffic_light/" + id + "/status"))
  , sub_(std::make_shared<mqtt_dataio::subscriber<fleet::LightControl>>("/traffic_light/" + id + "/control"
          , std::bind(&TrafficLight::on_control_message, this, std::placeholders::_1)))
{
}

TrafficLight::~TrafficLight()
{
}

void TrafficLight::update(const fleet::LightStatus &s)
{
    status_ = s;
    if(pub_)
    {
        fleet::RoadsideInfo msg;
        msg.set_id(id_);
        msg.set_lightstatus(s);
        msg.set_latitude(lat_);
        msg.set_longitude(lon_);
        msg.set_altitude(alt_);
        pub_->publish(msg);
    }
}

void TrafficLight::run_mqtt_client(const std::string &ip, const uint16_t &port, const std::string &user, const std::string &pwd)
{
    mqtt_dataio::client::ins().init(ip, port, user, pwd);
}

void TrafficLight::on_control_message(const fleet::LightControl &msg)
{
    if(cb_) cb_(id_, msg.lightstatus());
}

TrafficLightArray::TrafficLightArray(const std::string& topic_pub, const std::string& topic_control, const ControlCallback& cb)
  : topic_pub_(topic_pub)
  , topic_control_(topic_control)
  , cb_(cb)
  , pub_(std::make_shared<mqtt_dataio::publisher<fleet::RoadsideInfoList>>(topic_pub_))
  , sub_(nullptr)
{
}

bool TrafficLightArray::add_light(const std::string &id, const double lat, const double lon, const double alt)
{
    std::call_once(control_once_flag_, [this](){
        sub_ = (std::make_shared<mqtt_dataio::subscriber<fleet::LightControl>>(
            topic_control_, std::bind(&TrafficLightArray::on_control_message, this, std::placeholders::_1)));
    });
    auto itor = lights_.find(id);
    if(itor != lights_.end()) return false;
    lights_.emplace(id, std::make_shared<TrafficLight>(id, lat, lon, alt));
    return true;
}

bool TrafficLightArray::remove_light(const std::string &id)
{
    auto itor = lights_.find(id);
    if(itor == lights_.end()) return false;
    lights_.erase(itor);
    return true;
}

bool TrafficLightArray::update(const std::string &id, const fleet::LightStatus &s)
{
    auto itor = lights_.find(id);
    if(itor == lights_.end()) return false;
    itor->second->update(s);
    return true;
}

void TrafficLightArray::update(const fleet::LightStatus &s)
{
    for(auto itor = lights_.begin(); itor != lights_.end(); ++itor)
    {
        itor->second->update(s);
    }
}

bool TrafficLightArray::publish()
{
    if(pub_)
    {
        fleet::RoadsideInfoList msg;
        for(auto itor = lights_.begin(); itor != lights_.end(); ++itor)
        {
            auto light = msg.add_roadsidelist();
            light->set_id(itor->second->id_);
            light->set_longitude(itor->second->lon_);
            light->set_latitude(itor->second->lat_);
            light->set_altitude(itor->second->alt_);
            light->set_islight(1);
            light->set_lightstatus(itor->second->status_);
        }
        pub_->publish(msg);
        return true;
    }
    return false;
}

void TrafficLightArray::on_control_message(const fleet::LightControl &msg)
{
    auto itor = lights_.find(msg.id());
    if(itor != lights_.end() and cb_)
    {
        cb_(*itor->second, msg.lightstatus());
    }
}
