from flask import Flask, request
from flask.json import jsonify
from data.equipment.truck import TruckInfo
from data.equipment.excavator import ExcavatorInfo
from data.equipment.dump import DumpInfo
from core.dispatcher import PreSchedule
from core.group import Group
from flask_caching import Cache
from alg.algorithm import ExpectedTime
from core.submit import DispatchSubmission
from core.group import GroupDispatcher
from core.group import group_direct2redis
from core.util import POST
from util.area_analysis import *

config = {
    "DEBUG": True,          # some Flask specific configs
    "CACHE_TYPE": "SimpleCache",  # Flask-Caching related configs
    "CACHE_DEFAULT_TIMEOUT": 300
}

app = Flask(__name__)
app.config.from_mapping(config)
cache = Cache(app)


@app.route("/dispatch", methods=["POST"])
def dispatch_request():

    # 获取报文数据
    data_json = request.get_json()

    # 分组id
    group_id = data_json.get("group_id")

    # 调度开始时间
    rtd_start_time = datetime.now()

    # 初始化日志
    set_log()
    # 获取日志器
    logger = get_logger("zxt.request")

    # 更新周期参数
    logger.info("#####################################请求调度更新开始#####################################")

    try:

        # 清空数据库缓存
        session_mysql.commit()
        session_mysql.flush()

        # 清空数据库缓存
        session_postgre.commit()
        session_postgre.flush()
    except Exception as es:
        logger.error("数据库访问异常")
        logger.error(es)
        return jsonify(msg="数据库访问异常", code=501)

    try:

        # 全局参数更新
        global_period_para_update()
        # get_global_para_from_cache(cache)

    except Exception as es:
        logger.error("全局参数更新异常")
        logger.error(es)
        session_mysql.rollback()
        session_postgre.rollback()
        return jsonify(msg="全局参数更新异常", code=502)

    try:
        # 更新调度信息
        DispatchInfo.reset()

        DispatchInfo.update_device_group_structure()

        if group_id not in DispatchInfo.group_set:
            raise Exception("请求调度分组不存在")

        DispatchInfo.update_route_distance()

        DispatchInfo.update_group_mode()

        DispatchInfo.update_group_name()
    except Exception as es:
        logger.error("调度信息更新异常")
        logger.error(es)
        session_mysql.rollback()
        session_postgre.rollback()
        return jsonify(msg="调度信息更新异常", code=503)

    logger.info("Dispatchinfo，更新后信息")
    logger.info("group_set")
    logger.info(DispatchInfo.group_set)
    logger.info("group_excavator_dict")
    logger.info(DispatchInfo.group_excavator_dict)
    logger.info("group_unload_area_dict")
    logger.info(DispatchInfo.group_unload_area_dict)
    logger.info("group_truck_dict")
    logger.info(DispatchInfo.group_truck_dict)
    logger.info("group_mode")
    logger.info(DispatchInfo.group_mode)
    logger.info("load_distance")
    logger.info(DispatchInfo.load_distance)
    logger.info("unload_distance")
    logger.info(DispatchInfo.unload_distance)

    try:

        # 实例化设备对象
        dump = DumpInfo()
        excavator = ExcavatorInfo()
        truck = TruckInfo(dump, excavator)

        # 设备信息更新
        dump.dump_para_period_update()
        excavator.excavator_para_period_update()
        truck.truck_para_period_update(dump, excavator)
        truck.state_period_update()

        # 实例化调度预测器
        pre_sch = PreSchedule(truck, excavator, dump)

        # 实例化输出器
        submission = DispatchSubmission(dump, excavator, truck)

        # 实例化调度分组
        group = Group(group_id, truck, pre_sch, excavator, dump)

        # 更新调度分组信息
        group.info_update()

    except Exception as es:
        logger.error("对象实例化异常")
        logger.error(es)
        session_mysql.rollback()
        session_postgre.rollback()
        return jsonify(msg="对象实例化异常", code=504)

    try:

        # # 调度分组派车计划计算
        # try:
        #     truck_dispatch_plan_dict = group.group_dispatch(ExpectedTime)
        # except Exception as es:
        #     logger.error(es)
        #     logger.error(f'分组{group.group_id} 调度计算异常')
        #
        # try:
        #
        #     logger.info(f'调度分组: {group.group_id} {DispatchInfo.group_name[group.group_id]}')
        #     submission.group_dispatch_to_redis(group, truck_dispatch_plan_dict)
        # except Exception as es:
        #     logger.error(es)
        #     logger.error(f'分组{group.group_id} 调度写入异常')

        # 调度分组派车计划计算
        try:
            group_dispatcher = GroupDispatcher(group)

            truck_dispatch_plan_dict = group_dispatcher.group_dispatch(ExpectedTime)

            if truck_dispatch_plan_dict is None:
                logger.error(f'分组 {group.group_id} 调度异常')
        except Exception as es:
            logger.error(es)
            logger.error(f'分组{group.group_id} 调度计算异常')

        try:

            logger.info(f'调度分组: {group.group_id} {DispatchInfo.group_name[group.group_id]}')
            submission.group_dispatch_to_redis(group, truck_dispatch_plan_dict)
        except Exception as es:
            group_direct2redis(group)
            logger.error(es)
            logger.error(f'分组{group.group_id} 调度写入异常')


    except Exception as es:
        logger.error("最外层异常捕获")
        logger.error(es)
        return jsonify(msg="未知异常, 请联系管理员", code=505)

    session_mysql.close()
    session_postgre.close()

    logger.info("#####################################请求调度更新结束#####################################")

    # 调度结束时间
    rtd_end_time = datetime.now()

    print(f'调度时耗 {rtd_end_time - rtd_start_time}')

    return jsonify(msg="success", code=0)


@app.route("/go_through", methods=["POST"])
def redispatch_request():
    # 获取报文数据
    data_json = request.get_json()

    # 车辆id
    request_truck_id = data_json.get("truck_id")

    # request_truck_id = '0349fbdf-3c37-4fb3-867f-bea98a42af4a'

    # 调度开始时间
    rtd_start_time = datetime.now()

    # 初始化日志
    set_log()
    # 获取日志器
    logger = get_logger("zxt.Request")

    # 更新周期参数
    logger.info("#####################################请求调度更新开始#####################################")

    '''
    1. 更新全局参数信息
    '''

    try:

        # 清空数据库缓存
        session_mysql.commit()
        session_mysql.flush()

        # 清空数据库缓存
        session_postgre.commit()
        session_postgre.flush()
    except Exception as es:
        logger.error("数据库访问异常")
        logger.error(es)
        return jsonify(msg="未知异常, 请联系管理员", code=501)

    try:

        # 全局参数更新
        global_period_para_update()
        # get_global_para_from_cache(cache)

    except Exception as es:
        logger.error("全局参数更新异常")
        logger.error(es)
        session_mysql.rollback()
        session_postgre.rollback()
        return jsonify(msg="未知异常, 请联系管理员", code=502)

    try:
        # 更新调度信息
        DispatchInfo.reset()

        DispatchInfo.update_device_group_structure()

        DispatchInfo.update_route_distance()

        DispatchInfo.update_group_mode()

        DispatchInfo.update_group_name()
    except Exception as es:
        logger.error("调度信息更新异常")
        logger.error(es)
        session_mysql.rollback()
        session_postgre.rollback()
        return jsonify(msg="未知异常, 请联系管理员", code=503)

    logger.info("Dispatchinfo，更新后信息")
    logger.info("group_set")
    logger.info(DispatchInfo.group_set)
    logger.info("group_excavator_dict")
    logger.info(DispatchInfo.group_excavator_dict)
    logger.info("group_unload_area_dict")
    logger.info(DispatchInfo.group_unload_area_dict)
    logger.info("group_truck_dict")
    logger.info(DispatchInfo.group_truck_dict)
    logger.info("group_mode")
    logger.info(DispatchInfo.group_mode)
    logger.info("load_distance")
    logger.info(DispatchInfo.load_distance)
    logger.info("unload_distance")
    logger.info(DispatchInfo.unload_distance)

    try:

        # # 实例化设备对象
        dump = DumpInfo()
        excavator = ExcavatorInfo()
        truck = TruckInfo(dump, excavator)

        # 设备信息更新
        dump.dump_para_period_update()
        excavator.excavator_para_period_update()
        truck.truck_para_period_update(dump, excavator)
        truck.state_period_update()

    except Exception as es:
        logger.error("对象实例化异常")
        logger.error(es)
        session_mysql.rollback()
        session_postgre.rollback()
        return jsonify(msg="未知异常, 请联系管理员", code=504)

    '''
    2. 读取二次调度所需信息（车辆位置、分组、挖机等）
    '''

    try:

        truck_uuid_to_name_dict = get_value("truck_uuid_to_name_dict")

        # 获取请调车辆名
        request_truck_name = truck_uuid_to_name_dict[request_truck_id]

    except Exception as es:
        logger.error(es)
        return jsonify(msg="数据库异常, 车辆编号未知", code=510)

    try:
        # 读取请调车辆所属分组
        group_id = DispatchInfo.truck_group_dict[request_truck_id]

        # 读取分组挖机集合
        excavators_id = DispatchInfo.get_excavator(group_id)

        # 读取车辆位置信息
        truck_locates_dict = get_trucks_locate()

        logger.info("truck_locates_dict")
        logger.info(truck_locates_dict)

        with open(json_file, encoding='UTF-8') as f:
            load_value = json.load(f)

            postgre_config = load_value["postgresql"]

            closer_area_name = postgre_config["closer_area_name"]
            closer_area_id = session_postgre.query(DiggingWorkArea).filter_by(Name=closer_area_name).first().Id

            further_area_name = postgre_config["further_area_name"]
            further_area_id = session_postgre.query(DiggingWorkArea).filter_by(Name=further_area_name).first().Id

        logger.info("近端装载区id")
        logger.info(closer_area_id)
        logger.info("远端装载区id")
        logger.info(further_area_id)

        # 读取两个挖机id
        if closer_area_id in DispatchInfo.load_excavator_dict and further_area_id in DispatchInfo.load_excavator_dict:
            closer_excavator_id, further_excavator_id = DispatchInfo.load_excavator_dict[closer_area_id], \
                DispatchInfo.load_excavator_dict[further_area_id]
        else:
            return jsonify(msg="装载点信息错误", code=506)

        # 读取挖机状态
        closer_excavator_state, further_excavator_state = get_excavator_state(closer_excavator_id), \
                                                          get_excavator_state(further_excavator_id)

        # 读取两个装载区入场点id
        closer_entrance_node_id = session_postgre.query(DiggingWorkArea).filter_by(Id=closer_area_id).first().EntranceNodeId

        further_entrance_node_id = session_postgre.query(DiggingWorkArea).filter_by(
            Id=further_area_id).first().EntranceNodeId

        logger.info("近端装载区入场点")
        logger.info(closer_entrance_node_id)
        logger.info("远端装载区入场点")
        logger.info(further_entrance_node_id)

    except Exception as es:
        logger.error("读取装载区及车辆信息异常")
        logger.error(es)
        return jsonify(msg="未知异常, 请联系管理员", code=505)

    try:
        # 读取请调车辆所在路段信息
        request_truck_lane_id = truck_locates_dict[request_truck_id]

        logger.info("request_truck_lane_id:")
        logger.info(request_truck_lane_id)
        logger.info(truck_locates_dict)

    except Exception as es:
        logger.error(f'车辆 {request_truck_name} 位置信息不可用')
        logger.error(es)
        return jsonify(msg=f'车辆 {request_truck_name} 位置信息不可用, 请联系管理员', code=505)

    '''
    3. 调度判断逻辑
    '''
    # 车辆已驶过第一个装载点，车辆只能驶往远端装载区
    if not truck_pass_first_area(request_truck_id, request_truck_lane_id, closer_entrance_node_id, further_entrance_node_id):
        logger.info("车辆已经过近端装载区")
        target_excavator = DispatchInfo.load_excavator_dict[further_area_id]
        # truck_dispatch_to_redis(request_truck_id, group_id, DispatchInfo.load_excavator_dict[further_area_id])

    # 车辆未驶过第一个装载点，进入二次调度逻辑
    else:
        # 近端挖机空闲
        if closer_excavator_state == 0:
            logger.info("近端挖机空闲, 调度车辆前往")
            target_excavator = DispatchInfo.load_excavator_dict[closer_area_id]
            # truck_dispatch_to_redis(request_truck_id, group_id, DispatchInfo.load_excavator_dict[closer_area_id])
        # 远端挖机空闲
        elif further_excavator_state == 0:
            logger.info("远端挖机空闲, 调度车辆前往")
            target_excavator = DispatchInfo.load_excavator_dict[further_area_id]
            # truck_dispatch_to_redis(request_truck_id, group_id, DispatchInfo.load_excavator_dict[further_area_id])
        # 两挖机均不空闲
        else:
            # 两装载点间路段集合
            lane_set = get_lanes_between_entrances(closer_entrance_node_id, further_entrance_node_id)

            # 选择合适装载区
            target_excavator = area_choose(excavators_id, closer_area_id, further_area_id, lane_set, logger, truck, truck_locates_dict)

    # 派车计划写入redis
    truck_dispatch_to_redis(request_truck_id, group_id, target_excavator)

    POST(request_truck_id)

    # except Exception as es:
    #     logger.error(" ")
    #     logger.error(es)
    #     session_mysql.rollback()
    #     session_postgre.rollback()
    #     return jsonify(msg="未知异常, 请联系管理员", code=504)

    logger.info("#####################################请求调度更新结束#####################################")

    # 调度结束时间
    rtd_end_time = datetime.now()

    print(f'调度时耗 {rtd_end_time - rtd_start_time}')

    return jsonify(msg="success", code=0)

if __name__ == "__main__":
    app.run(host='0.0.0.0', port=80)