/*
 * @Descripttion: 
 * @version: 
 * @Author: wxin
 * @Date: 2023-11-07 01:52:07
 * @email: xin.wang@waytous.com
 * @LastEditors: wxin
 * @LastEditTime: 2023-11-23 02:22:56
 */


#include "vision_transform/ros_visualization.h"



RosVisualization::RosVisualization(ros::NodeHandle &nodehandle, std::string frame_id){
    m_nodehandle = nodehandle;
    current_frame_id = frame_id;
    // m_pubCloudPoint_3D  = m_nodehandle.advertise<sensor_msgs::PointCloud2>("vision_points_3d", 5);
    m_pubCloudPoint_BEV = m_nodehandle.advertise<sensor_msgs::PointCloud2>("/perception/camera/vision_points_bev", 5);
    m_pub_bounding_boxes = m_nodehandle.advertise<jsk_recognition_msgs::BoundingBoxArray>("/perception/camera/bounding_boxes", 5);
    m_pub_image = m_nodehandle.advertise<sensor_msgs::Image>("/perception/camera/vision_image_result", 1);
}


void RosVisualization::Visualization(BEV_GridMap& vGrid, cv::Mat& image, const std::vector<object_Seg_info> &objs){

    pcl::PointCloud<pcl::PointXYZRGB> lidar_acc_cloud;  //累计目标点云
    if(objs.empty())
    {
        ShowCloudPoints(lidar_acc_cloud);
        ShowBoundingBoxArray(vGrid.vInsObjs); 
        ShowImage(image, objs); 
        return;   
    }
    lidar_acc_cloud.points.reserve(200*200); // get 200x200 points at most
    vGrid.initVisualInfo();
    vGrid.drawInsMasks();      
    vGrid.drawInsRects();

    changeImageToBev(vGrid.m_imgBEV_static, vGrid.m_cloudBEV_static);
    for( auto&  pt : vGrid.m_cloudBEV_static){
        if(pt.z <= 0) continue;
        // pt.y += yshift;
        lidar_acc_cloud.points.emplace_back(pt);
    }
    changeImageToBev(vGrid.m_imgBEV_Instance, vGrid.m_cloudBEV_Instance);
    for( auto&  pt : vGrid.m_cloudBEV_Instance){
        if(pt.z <= 0) continue;
        lidar_acc_cloud.points.emplace_back(pt);
    }

    lidar_acc_cloud.width = lidar_acc_cloud.points.size();
    lidar_acc_cloud.height = 1;
    ShowCloudPoints(lidar_acc_cloud);
    ShowBoundingBoxArray(vGrid.vInsObjs); 
    ShowImage(image, objs);
    return;
};


void RosVisualization::ShowImage(cv::Mat& image, const std::vector<object_Seg_info>& objs){
    // 图像可视化
    for(auto& obj: objs){
        cv::Scalar color =  obj.valid > 0 ? obj.color : cv::Scalar(255, 255, 255);
        if(obj.valid == 2){ // from depth estimnation
            color = cv::Scalar(color[0] / 2, color[1] / 2, color[2] / 2);
        }
        cv::putText(image, obj.name + ":" + waytous::deepinfer::common::formatValue(obj.fConfidence, 2), 
                    cv::Point(obj.rc.x, obj.rc.y - 5), 
                    0, 0.6, color, 2, cv::LINE_8);
        auto& rt = obj.rc;
        cv::rectangle(image, rt, color, 2);
        cv::Mat instance_mask = obj.mask;
        for (int x = rt.x; x < rt.x + rt.width; x++) {
            for (int y = rt.y; y < rt.y + rt.height; y++) {
                float val = instance_mask.at<uchar>(y - rt.y, x - rt.x);
                if (val <= 0.5) continue;
                // LOG_INFO<<val<<" ";
                image.at<cv::Vec3b>(y, x)[0] = image.at<cv::Vec3b>(y, x)[0] / 2 + color[0] / 2;
                image.at<cv::Vec3b>(y, x)[1] = image.at<cv::Vec3b>(y, x)[1] / 2 + color[1] / 2;
                image.at<cv::Vec3b>(y, x)[2] = image.at<cv::Vec3b>(y, x)[2] / 2 + color[2] / 2;
            }
        }
    }

    sensor_msgs::ImagePtr msg = cv_bridge::CvImage(std_msgs::Header(), "bgr8", image).toImageMsg();
    m_pub_image.publish(msg);
}



int RosVisualization::ShowBoundingBoxArray(const std::vector<InstanceBevObject>& bevObjs)
{
    // box fitting
    jsk_recognition_msgs::BoundingBoxArray arraybox;

    for (auto& obj: bevObjs)
    {
        jsk_recognition_msgs::BoundingBox box;
        box.header.stamp = ros::Time::now();
        box.header.frame_id = current_frame_id;
        cv::Point2f pos =  BEV_GridMap::convertmap_2_3d((int)obj.rrc.center.x, (int)obj.rrc.center.y);
        box.pose.position.x = pos.x;
        box.pose.position.y = pos.y;
        box.pose.position.z = 0;
        auto orient = tf::createQuaternionFromYaw((-obj.rrc.angle+90)* 3.14 / 180);
        box.pose.orientation.x = orient.x();
        box.pose.orientation.y = orient.y();
        box.pose.orientation.z = orient.z();
        box.pose.orientation.w = orient.w();
        box.dimensions.x = obj.rrc.size.width*BEV_GRID_RESOLUTION;
        box.dimensions.y = obj.rrc.size.height*BEV_GRID_RESOLUTION;
        box.dimensions.z = 0.1;
        box.label = obj.color.rgba;
        //filter object in the air
        arraybox.boxes.emplace_back(box);

    }
    // publish
    arraybox.header.stamp = ros::Time::now();
    arraybox.header.frame_id = current_frame_id;
    m_pub_bounding_boxes.publish(arraybox);

    return 0;
}


template <typename PointT>
void RosVisualization::ShowCloudPoints(const pcl::PointCloud<PointT> &pt_cloud)
{
    sensor_msgs::PointCloud2 map_point_pub;
    pcl::toROSMsg(pt_cloud, map_point_pub);
    map_point_pub.header.frame_id = current_frame_id;
    map_point_pub.header.stamp = ros::Time::now();
    m_pubCloudPoint_BEV.publish(map_point_pub);
}




