#include "vision_transform/ComFunction.h"
#include <unistd.h>
#include<sys/stat.h>


/**
 * 按列连接距离较近的间隙
 * binImage: [in] 二值图像
 * nLength:  [in] 连接长度阈值
 * bConnectEdge: [in] 是否处理边界部分
*/
void ConnectSmallHeightGap(cv::Mat &binImage, int nLength, int bConnectEdge)
{
	int nWidth = binImage.cols;
	int nHeight = binImage.rows;
	int start = 0;
	int end;
	int i, j;
	int bFlag;

	for (i = 0; i < nWidth; ++i)
	{
		bFlag = 0;
		for (j = 0; j < nHeight; ++j)
		{
			auto& pbySrc = binImage.at<unsigned char>(j, i);
			if (bFlag && (pbySrc != 0 || j == nHeight - 1))
			{
				end = j;
				bFlag = 0;
				if (bConnectEdge || (!bConnectEdge && start != 0 && end != nHeight - 1))
				{
					if ((end - start) <= nLength)
					{
						for(int m = start; m < end; m++ ){
							binImage.at<unsigned char>(m, i) = 255;
						}
					}
				}
			}

			if (!bFlag && pbySrc== 0)
			{
				start = j;
				bFlag = 1;
			}
		}
	}
}

/*******************************************************************
 *函数名称：Findlabel
 *输入参数：label    经过连通域检测之后的矩阵
 *        number   被标记的数字号
 *输出参数：vvPt     存储被标记联通域点的坐标
 *函数说明：查找连通域中指定标记数的坐标
 ********************************************************************/
int Findlabel(const cv::Mat &label, int number, std::vector<std::vector<cv::Point>> &vvPt)
{
	vvPt.clear();
	std::vector<cv::Point> vpt;
	for (int i = 0; i <= number; i++)
	{
		vvPt.push_back(vpt);
	}

	int width = label.cols;
	int height = label.rows;

	if (CV_16U == label.type())
	{
		for (int j = 0; j < height; j++)
		{
			auto pDest = label.ptr<unsigned short>(j);
			for (int i = 0; i < width; i++)
			{
				if (*pDest)
				{
					vvPt[*pDest].push_back(cv::Point(i, j));
				}
				pDest++;
			}
		}
	}
	else if (CV_32S == label.type())
	{
		for (int j = 0; j < height; j++)
		{
			auto pDest = label.ptr<int>(j);
			for (int i = 0; i < width; i++)
			{
				if (*pDest)
				{
					vvPt[*pDest].push_back(cv::Point(i, j));
				}
				pDest++;
			}
		}
	}

	return 0;
}


cv::Rect GetBundingRect(std::vector<cv::Point> &vPt)
{
	if (vPt.empty())
		return cv::Rect(0, 0, 0, 0);

	int xMax, xMin, yMax, yMin;
	xMax = xMin = vPt[0].x;
	yMax = yMin = vPt[0].y;

	for (int i = 0; i < vPt.size(); i++)
	{
		xMax = MAX(xMax, vPt[i].x);
		xMin = MIN(xMin, vPt[i].x);
		yMax = MAX(yMax, vPt[i].y);
		yMin = MIN(yMin, vPt[i].y);
	}

	return cv::Rect(xMin, yMin, xMax - xMin+1, yMax - yMin+1);
}


void removeSmallArea(cv::Mat &imageBin, int nLen)
{
	std::vector<std::vector<cv::Point>> vvBinPt;
	cv::Mat bin = imageBin.clone();
	cv::Mat labels;
	int n_comps = connectedComponents(bin, labels, 8, CV_16U);

	Findlabel(labels, n_comps, vvBinPt);

	for (int i = 1; i < n_comps; i++)
	{
		std::vector<cv::Point> &vBinPt = vvBinPt[i];
		if (vBinPt.size() == 0)
			continue;

		cv::Rect rect = GetBundingRect(vBinPt);
		if (rect.width < nLen) // 小目标
		{
			for (int m = 0; m < vBinPt.size(); m++)
			{
				imageBin.at<uchar>(vBinPt[m].y, vBinPt[m].x) = 0;
				//*(pDst + vBinPt[m].y*pImageBin->widthStep + vBinPt[m].x) = 0;
			}
		}
	}
}


/**
 * 区域生长法分割图像
 * mat [in] 原始图像
 * seedThr [in] 种子点阈值  将 edgeThr<= 像素值<=nMaxvalue的值当作种子点
 * edgeThr [in] 区域生长边界阈值 分割区域内像素值  >=edgeThr
 * imgBin [out] 分割的二值图像
 * nMaxvalue [in] 分割像素值的最大阈值  超过此阈值的点将不被分割
*/
int AreaGrow_segment(cv::Mat& mat, int seedThr, int edgeThr, cv::Mat & imgBin, int nMaxvalue)
{
	imgBin = cv::Mat::zeros(mat.size(), CV_8UC1);    //生长区域
 
	cv::Point waitSeed;    //待生长种子点
	int waitSeed_value = 0;    //待生长种子点像素值
	int opp_waitSeed_value = 0;   //mat_thresh中对应待生长种子点处的像素值
	
	std::vector<cv::Point> seedVector;     //种子栈
	/////// 根据 seedThr 和 nMaxvalue 值提取种子点
	for(int j = 0; j < mat.rows; j++){
		for(int i = 0; i < mat.cols; i++){
			if( mat.at<uchar>(j,i) >= seedThr && mat.at<uchar>(j,i) <= nMaxvalue  )
			{
				imgBin.at<uchar>(j,i) = 255;
				seedVector.push_back(cv::Point(i,j));
			}
		}
	}

	/////// 根据edgeThr进行区域生长
	//int direct[4][2] = { {0,-1},{1,0}, {0,1}, {-1,0} };   //4邻域,应该用4邻域减小时间复杂度
	int direct[8][2] = { {-1,-1}, {0,-1}, {1,-1}, {1,0}, {1,1}, {0,1}, {-1,1}, {-1,0} };  //8邻域

	while (!seedVector.empty())     //种子栈不为空则生长，即遍历栈中所有元素后停止生长
	{
		cv::Point seed = seedVector.back();     //取出最后一个元素
		seedVector.pop_back();         //删除栈中最后一个元素,防止重复扫描
		for (int i = 0; i < 8; i++)    //遍历种子点的4邻域
		{
			waitSeed.x = seed.x + direct[i][0];    //第i个坐标0行，即x坐标值
			waitSeed.y = seed.y + direct[i][1];    //第i个坐标1行，即y坐标值

			//检查是否是边缘点
			if (waitSeed.x < 0 || waitSeed.y < 0 ||
				waitSeed.x >(mat.cols - 1) || (waitSeed.y > mat.rows - 1))
				continue;

			waitSeed_value = imgBin.at<uchar>(cv::Point(waitSeed.x, waitSeed.y));   //为待生长种子点赋对应位置的像素值
			opp_waitSeed_value = mat.at<uchar>(cv::Point(waitSeed.x, waitSeed.y));
			if (waitSeed_value == 0)     //判断waitSeed是否已经被生长，避免重复生长造成死循环
			{
				if (opp_waitSeed_value >= edgeThr && opp_waitSeed_value <= nMaxvalue )     //区域生长条件
				{
					imgBin.at<uchar>(cv::Point(waitSeed.x, waitSeed.y)) = 255;
					seedVector.push_back(waitSeed);    //将满足生长条件的待生长种子点放入种子栈中
				}
			}
		}
	}

	return 0;
}


/**
 * 填充二值区域中的空洞
 * srcBw： [in]  输入图像
 * dstBw： [in/out] 输出图像
 * sz: [in] 小于sz尺寸的空洞将被填充
*/
void fillHole(const cv::Mat& srcBw, cv::Mat &dstBw, cv::Size sz)
{
    cv::Size m_Size = srcBw.size();
	int nExtend = 4;
    cv::Mat Temp = cv::Mat::zeros(m_Size.height+nExtend,m_Size.width+nExtend,srcBw.type());//延展图像
    srcBw.copyTo(Temp(cv::Range(nExtend/2, m_Size.height + nExtend/2), cv::Range(nExtend/2, m_Size.width + nExtend/2)));
 
    cv::floodFill(Temp, cv::Point(0, 0), cv::Scalar(255));
 
    cv::Mat cutImg;//裁剪延展的图像
    Temp(cv::Range(nExtend/2, m_Size.height + nExtend/2), cv::Range(nExtend/2, m_Size.width + nExtend/2)).copyTo(cutImg);
	
	cv::Mat fill = ~cutImg;
	//removeLargeArea( fill, sz );
    dstBw = srcBw | fill;
}


/**
 * @name: changeImageToBev
 * @msg: 转换图像颜色到RGB点云    图像和点云空间需要预先分配好
 * @param {Mat&} imageRgb      输入图像
 * @param {PointCloud<PointXYZRGB>&} pointBEV  输入/输出点云
 * @return {*}
 */
int changeImageToBev(const cv::Mat& imageRgb, pcl::PointCloud<pcl::PointXYZRGB>& pointBEV)
{
    int count = 0;
    for(int j = 0; j < imageRgb.rows; j++)
    {
        for(int i = 0; i < imageRgb.cols; i++)
        {
            auto& rgb = imageRgb.at<cv::Vec3b>(j,i);    
            if( rgb[0] || rgb[1] || rgb[2]){
                pointBEV[count].r = rgb[0];
                pointBEV[count].g = rgb[1];
                pointBEV[count].b = rgb[2];
				pointBEV[count].z = 0.1;
            }
            count++;
        }
    }
    return 0;
}


int  KeepThrLargestAreaRect(cv::Mat& imgBin)
{    
    cv::Mat temp;
    cv::Mat labels;
    imgBin.copyTo(temp);

    //1. 标记连通域
    int n_comps = connectedComponents(temp, labels, 4, CV_16U);
    std::vector<int> histogram_of_labels(n_comps, 0);
    // for (int i = 0; i < n_comps; i++)//初始化labels的个数为0
    // {
    //     histogram_of_labels.push_back(0);
    // }

    int rows = labels.rows;
    int cols = labels.cols;
    for (int row = 0; row < rows; row++) //计算每个labels的个数
    {
        for (int col = 0; col < cols; col++)
        {
            histogram_of_labels.at(labels.at<unsigned short>(row, col)) += 1;
        }
    }
    histogram_of_labels.at(0) = 0; //将背景的labels个数设置为0

    //2. 计算最大的连通域labels索引
    int maximum = 0;
    int max_idx = 0;
    for (int i = 0; i < n_comps; i++)
    {
        if (histogram_of_labels.at(i) > maximum)
        {
            maximum = histogram_of_labels.at(i);
            max_idx = i;
        }
    }


    //3. 将最大连通域外的点置0 
    for (int row = 0; row < rows; row++) 
    {
        for (int col = 0; col < cols; col++)
        {
            if (labels.at<unsigned short>(row, col) != max_idx  && imgBin.at<uchar>(row,col) )
            {
                imgBin.at<uchar>(row,col) = 0;
            }
        }
    }

    return 0;
}

