/*
 * @Descripttion: 
 * @version: 
 * @Author: wxin
 * @Date: 2023-09-03 01:53:20
 * @email: xin.wang@waytous.com
 * @LastEditors: wxin
 * @LastEditTime: 2023-11-22 07:01:31
 */

#ifndef SENSORMANAGER_H
#define SENSORMANAGER_H

#include <string>
#include <boost/thread/thread.hpp>

#include <opencv2/opencv.hpp>
#include <pcl/point_cloud.h>
#include <pcl/point_types.h>
#include <pcl/common/transforms.h>
#include <pcl_conversions/pcl_conversions.h>

#include <interfaces/base_model.h>
#include <libs/ios/detection.h>
#include <libs/ios/camera_ios.h>

#include <sensor_msgs/Image.h>
#include <sensor_msgs/PointCloud2.h>
#include <geometry_msgs/Pose.h>
#include <geometry_msgs/Vector3.h>
#include <geometry_msgs/Polygon.h>
// msg
#include "waytous_perception_msgs/ObjectArray.h"
#include "waytous_perception_msgs/Object.h"
#include "waytous_perception_msgs/Rect.h"

#include "common_maps.h"
#include "utils.h"
#include "camera_frame.h"
#include "undistort/undistort_v2.h"
#include "vision_transform/ros_visualization.h"

typedef pcl::PointCloud<pcl::PointXYZI> Cloud;
typedef Cloud::ConstPtr CloudConstPtr;

namespace perception::camera{


class sensorManager
{
public:
    /**
     * @name: sensorManager
     * @msg: 构造函数
     * @param {ImageInferNodeParam&} param 参数
     * @return {*}
     */
    sensorManager(ros::NodeHandle& nodehandle, ImageInferNodeParam& param);

    ~sensorManager(){
        delete pub_thread_;
        pub_thread_ = nullptr;
    }

    /**
     * @name: infer
     * @msg: 模型推理
     * @return {cameraFramePtr} 推理结果
     */
    cameraFramePtr infer();

    /**
     * @name: updateImage
     * @msg: 更新订阅的图像数据
     * @param {sensor_msgs::ImageConstPtr} img_msg 图像数据
     * @return {*}
     */
    void updateImage(sensor_msgs::ImageConstPtr img_msg){
        {
            std::lock_guard<std::mutex> lock_(image_mutex);
            img_timestamp_sys_ = GetMillSecondTimeStamp();
            current_image_ = img_msg;
        }
    };
    
    /**
     * @name: getCurImage
     * @msg: 获取最新的图像数据
     * @param {*}
     * @return {sensor_msgs::ImageConstPtr} 图像数据指针
     */
    sensor_msgs::ImageConstPtr getCurImage(){
        {
            std::lock_guard<std::mutex> lock_(image_mutex);
            if(current_image_ == nullptr){
                return nullptr;
            }
            auto temp = current_image_;
            current_image_ = nullptr;
            std::cout<<"image waiting time: " << GetMillSecondTimeStamp() - img_timestamp_sys_ << "ms" << std::endl;
            return temp;
        }
    };

    /**
     * @name: updatePointcloud
     * @msg: 更新点云信息
     * @param {sensor_msgs::PointCloud2ConstPtr>} pdata
     * @return {*}
     */
    void updatePointcloud(sensor_msgs::PointCloud2ConstPtr pdata){
        {
            std::lock_guard<std::mutex> lock_(pointcloud_mutex);
            pc_timestamp_sys_ = GetMillSecondTimeStamp();
            current_pointcloud_ = pdata;
        }
    };


    /**
     * @name: getCurPointcloud
     * @msg: 获取最新的点云数据
     * @param {*}
     * @return {sensor_msgs::PointCloud2ConstPtr} 点云数据指针
     */
    sensor_msgs::PointCloud2ConstPtr getCurPointcloud(){
        {
            std::lock_guard<std::mutex> lock_(pointcloud_mutex);
            auto temp = current_pointcloud_;
            int64_t tm_now = GetMillSecondTimeStamp();
            // 超过200ms的点云丢弃
            if(std::abs(tm_now - pc_timestamp_sys_) > pc_timevalid_interval_){
                current_pointcloud_ = nullptr;
            }
            return temp;
        }
    }


    /**
     * @name: publishDetection
     * @msg: 发布交通障碍物数据
     * @return {*}
     */
    void publishDetection();

    /**
     * @name: getLatestTimestamp
     * @msg: 获取最新数据的时间戳
     * @return {int64_t}
     */
    inline int64_t getLatestTimestamp(){
        std::lock_guard<std::mutex> lock_(ts_mutex);
        return latest_timestamp_;
    }


private:
    // 节点需要的参数
    ImageInferNodeParam infer_param;

    ros::Publisher obj_publisher_; // 交通障碍物发布者
    ros::Subscriber image_subscriber_; // 图像订阅者
    ros::Subscriber pointcloud_subscriber_; // 点云订阅者
    boost::thread* pub_thread_; // 发布线程
    
    std::mutex image_mutex; // 管理图像信息的互斥量
    std::mutex pointcloud_mutex; // 管理点云信息的互斥量
    std::mutex ts_mutex; // 管理时间戳的互斥量

    int64_t img_timestamp_sys_ = 0;  //接收到图像消息的系统时间
    int64_t pc_timestamp_sys_ = 0;  //接收到点云消息的系统时间
    int64_t pc_timevalid_interval_ = 200;  //点云帧有效时间, ms

    // timestamp: ms
    int64_t latest_timestamp_ = 0; // 最新数据时间戳
    int64_t frame_num_count = 0; // 程序启动到目前发布的数据量帧数

    // 推理模型指针
    std::shared_ptr<waytous::deepinfer::interfaces::BaseModel> multi_model = nullptr;
    // 2d转bev类
    std::shared_ptr<Visione_3D> vision_converter_ = nullptr;
    // 可视化类
    std::shared_ptr<RosVisualization> ros_visualization_ptr_ = nullptr;
    // 模型是否初始化成功
    bool inited = false;

    // 图像和点云数据指针
    sensor_msgs::ImageConstPtr current_image_ = nullptr;
    sensor_msgs::PointCloud2ConstPtr current_pointcloud_ = nullptr;

};




}


#endif  //SENSORMANAGER_H
