/*
 * @Descripttion: 
 * @version: 
 * @Author: wxin
 * @Date: 2023-09-03 02:18:07
 * @email: xin.wang@waytous.com
 * @LastEditors: wxin
 * @LastEditTime: 2023-11-22 09:22:01
 */


#include "sensor_manager.h"
#include <Eigen/Core>
#include <Eigen/Geometry>


namespace perception::camera{

/**
 * @name: sensorManager
 * @msg: 构造函数
 * @param {ImageInferNodeParam&} param 参数
 * @return {*}
 */
sensorManager::sensorManager(ros::NodeHandle& nodehandle, ImageInferNodeParam& param)
{

    infer_param = param;
    // last_header_ = eon::proto::common::Header();
    // 设置模型读取的路径
    waytous::deepinfer::common::ConfigRoot::SetRootPath(param.config_root_path);
    std::cout << "Set model's root path " << waytous::deepinfer::common::ConfigRoot::GetRootPath();
    // 设置推理的device编号
    CUDA_CHECK(cudaSetDevice(param.device));
    // 模型初始化
    multi_model.reset(waytous::deepinfer::interfaces::BaseModelRegisterer::GetInstanceByName("CameraModel"));
    if(!multi_model->Init(param.rgb_config_path)){
        std::cout<< "Error: init model error." <<std::endl;
        return;
    };
    // 初始化相机参数
    std::string param_name = "cameraParam";
    multi_model->modelUnitMap->SetIOPtr(param_name, infer_param.cameraParam_);
    // 2d to bev 类
    vision_converter_ = std::make_shared<Visione_3D>(param);
    // 可视化类
    if(param.vis_res){
        ros_visualization_ptr_ = std::make_shared<RosVisualization>(nodehandle, param.vis_frame_id);
    }
    inited = true;

    // 初始化订阅、发布话题
    image_subscriber_ = nodehandle.subscribe<sensor_msgs::Image>(param.rgb_sub_topic_name, 1, boost::bind(&sensorManager::updateImage, this, _1));
    pointcloud_subscriber_ = nodehandle.subscribe<sensor_msgs::PointCloud2>(param.pointcloud_sub_topic_name, 1, boost::bind(&sensorManager::updatePointcloud, this, _1));
    obj_publisher_ = nodehandle.advertise<waytous_perception_msgs::ObjectArray>(param.detection_pub_topic_name, 1);

    pub_thread_ = new boost::thread(boost::bind(&sensorManager::publishDetection, this));
};


/**
 * @name: infer
 * @msg: 模型推理
 * @return {cameraFramePtr} 推理结果
 */
cameraFramePtr sensorManager::infer()
{
    if(!inited){
        std::cout<< "Error: manager haven't init right."<<std::endl;
        return nullptr;
    }
    if(multi_model == nullptr)
    {
        std::cout<< "Error: manager init but multi model engine havenot init."<<std::endl;
        return nullptr;
    }
    auto ros_img = getCurImage();
    if(ros_img == nullptr){
        // std::cout<< "Warning: dont get image or current image frame got same timestamp with the pre one."<<std::endl;
        return nullptr;
    }
    // 图像转为opencv-mat
    cv::Mat rgb = cv_bridge::toCvShare(ros_img, "bgr8")->image;
    if(rgb.empty()){
        std::cout<<"Warning: current RGB image convert from proto.Image to cvImage with empty data."<<std::endl;
        return nullptr;
    }

    cameraFramePtr cptr = std::make_shared<cameraFrame>();
    // 定义模型的输入输出vector
    std::vector<cv::Mat*> inputs;
    std::vector<waytous::deepinfer::interfaces::BaseIOPtr> outputs;
    std::string cameraName = "rgb";

    clock_t start_total = clock();
    // 存入需要处理的图像
    inputs.push_back(&rgb);

    // 模型推理
    {
        // std::lock_guard<std::mutex> lock_(infer_mutex);
        clock_t start_ = clock();
        // 模型推理
        if(!multi_model->Exec(inputs, outputs)){
            std::cout << "Warning: current frame, multi model doesnot infer correct." << std::endl;
            return nullptr;
        };
        std::cout<<" model inference: "<<double(clock()-start_)/CLOCKS_PER_SEC<<"s"<<std::endl; 
        
        // 获取畸变矫正图像
        // auto undistort_img = std::dynamic_pointer_cast<waytous::deepinfer::ios::CameraSrcOut>(outputs[3])->img_ptr_->toCVMat().clone();
        auto uImg = std::dynamic_pointer_cast<waytous::deepinfer::ios::CameraSrcOut>(outputs[3])->img_ptr_;
        cv::Mat undistort_img(uImg->rows(), uImg->cols(), CV_8UC3);
        memcpy(undistort_img.data, uImg->cpu_data(), uImg->total() * sizeof(uint8_t));
        cptr->setImageSrc(undistort_img);
        // 获取语义分割结果
        cptr->setSemantics(std::dynamic_pointer_cast<waytous::deepinfer::ios::Semantics>(outputs[1]));
        // 获取实例分割结果
        cptr->setDetections(std::dynamic_pointer_cast<waytous::deepinfer::ios::Detection2Ds>(outputs[0])); 
        // 获取深度估计结果
        cptr->setDepth(std::dynamic_pointer_cast<waytous::deepinfer::ios::Depth>(outputs[2]));

        // 获取时间戳、sensor_header等信息
        auto header = ros_img->header;
        int64_t timestamp = header.stamp.sec * 1e3 + header.stamp.nsec / 1e6; // to ms
        cptr->timestamp_ = timestamp;
        cptr->timestamp_sys_ = GetMillSecondTimeStamp();
        cptr->sensor_header = header;
        cptr->cameraName = cameraName;
    }
    std::cout<<" infer total: "<<double(clock()-start_total)/CLOCKS_PER_SEC<<"s"<<std::endl; 
    return cptr;
};


/**
 * @name: publishDetection
 * @msg: 发布交通障碍物数据
 * @param {BEV_GridMap&} gmap 包含2d转换到bev的信息
 * @param {cameraFramePtr} frame_ptr 当前最新推理结果
 * @return {*}
 */
void sensorManager::publishDetection()
{
    while(true){
        clock_t start_ = clock();
        auto frame_ptr = infer();
        if(frame_ptr == nullptr){
            usleep(5000); // 5ms
            continue;
        }

        auto vGrid = std::make_shared<BEV_GridMap>();
        auto point_cloud_ptr = getCurPointcloud();
        if(point_cloud_ptr != nullptr){
            Cloud::Ptr pcl_cloud(new Cloud);
            pcl::fromROSMsg(*point_cloud_ptr, *pcl_cloud);
            // 2d->bev，结果保存到vGrid中
            // vision_converter_->Process_objects(frame_ptr->img_src_, pcl_cloud, frame_ptr->objs_, *vGrid);
            vision_converter_->Process_objects(frame_ptr->img_src_, frame_ptr->depth_, pcl_cloud, frame_ptr->objs_, *vGrid);
        }else{
            vision_converter_->Process_objects(frame_ptr->img_src_, frame_ptr->depth_, frame_ptr->objs_, *vGrid);
        }

        waytous_perception_msgs::ObjectArray obFrame;
        obFrame.sensor_type = waytous_perception_msgs::ObjectArray::SENSOR_CAMERA;

        int8_t obj_id = 0;
        for(auto& obj: vGrid->vInsObjs)
        {
            waytous_perception_msgs::Object ob;

            //int8 id = 1;   
            ob.id = obj_id++;

            //int8 property = 2;

            //float exist_prob = 3;  
            float prob = std::sqrt(obj.fConfidence);
            ob.exist_prob = prob;
            
            //waytous_perception_msgs/Rect obb2d = 4; 
            ob.obb2d.x = obj.rc.x;
            ob.obb2d.y = obj.rc.y;
            ob.obb2d.w = obj.rc.width;
            ob.obb2d.h = obj.rc.height;

            //geometry_msgs/Pose pose = 5;
            cv::Point2f pos =  BEV_GridMap::convertmap_2_3d((int)obj.rrc.center.x, (int)obj.rrc.center.y);
            double dangle = (90-obj.rrc.angle)* 3.14159 / 180;
            ::Eigen::Vector3d ea0(0,0,dangle);
            ::Eigen::Matrix3d matR;
            matR  = ::Eigen::AngleAxisd(ea0[0], ::Eigen::Vector3d::UnitX())
                                    * ::Eigen::AngleAxisd(ea0[1], ::Eigen::Vector3d::UnitY())
                                    * ::Eigen::AngleAxisd(ea0[2], ::Eigen::Vector3d::UnitZ());
            ::Eigen::Quaterniond orient;  
            orient = matR;

            ob.pose.position.x = pos.x;
            ob.pose.position.y = pos.y;
            ob.pose.position.z = 0;
            ob.pose.orientation.x = orient.x();
            ob.pose.orientation.y = orient.y();
            ob.pose.orientation.z = orient.z();
            ob.pose.orientation.w = orient.w();

            //geometry_msgs/Vector3   dimensions = 6
            ob.dimensions.x =  (obj.rrc.size.width*BEV_GRID_RESOLUTION);
            ob.dimensions.y = (obj.rrc.size.height*BEV_GRID_RESOLUTION);
            ob.dimensions.z = 1;
            
            //geometry_msgs/Polygon convex_hull = 7
            cv::Point2f vertices[4];
            obj.rrc.points(vertices);
            for(auto& v : vertices){
                geometry_msgs::Point32 p;
                auto pt3d = BEV_GridMap::convertmap_2_3d(v.x, v.y);
                p.x = pt3d.x;
                p.y = pt3d.y;
                p.z = 0;
                ob.convex_hull.points.emplace_back(p);
            }
            
            //sensor_msgs/PointCloud2 cloud = 8
            // TODO
            // ob.cloud.height = obj.vPoints.size();
            pcl::PointCloud<pcl::PointXYZ>::Ptr res_pcl_cloud(new pcl::PointCloud<pcl::PointXYZ>);
            res_pcl_cloud->points.reserve(obj.vPoints.size());
            for( auto& pt :obj.vPoints){
                auto pt3d = BEV_GridMap::convertmap_2_3d(pt.x, pt.y);
                pcl::PointXYZ pxyz;
                pxyz.x = pt3d.x;
                pxyz.y = pt3d.y;
                pxyz.z = 0;
                res_pcl_cloud->points.emplace_back(pxyz);
            }
            pcl::toROSMsg(*res_pcl_cloud, ob.cloud);

            // int8  type = 9
            ob.type = (name2type.find(obj.names) == name2type.end() ? Perception_Default_Classification : name2type[obj.names]);

            //float32[]  type_probs = 10
            std::vector<float> probs(waytous_perception_msgs::Object::TYPE_UNKNOWN_STATIC + 1, 0); // main-type length
            probs[ob.type] = prob;
            ob.type_probs = probs;

            // float32[] sub_type_probs = 11
            ob.sub_type = (name2subtype.find(obj.names) == name2subtype.end() ? Perception_Default_SubClassification : name2subtype[obj.names]);

            //repeated float sub_type_probs = 12;   
            std::vector<float> sub_probs(waytous_perception_msgs::Object::SUBTYPE_UNKNOWN_STATIC + 1, 0); // main-type length
            sub_probs[ob.sub_type] = prob;
            ob.sub_type_probs = sub_probs;

            // ############### tracking info
            // int8                                            track_id                # required
            // geometry_msgs/Vector3                           velocity                # required
            // float32[]                                       velocity_covariance     # optional
            // geometry_msgs/Vector3                           accelerate              # required
            // geometry_msgs/Vector3[]                         trace                   # optional
            // geometry_msgs/Vector3[]                         trajectory              # optional
            // FleetObject                                     fleet_object            # optional    
            obFrame.foreground_objects.emplace_back(ob);
        }
        obj_publisher_.publish(obFrame);
        std::cout<<" onRun :  "<<double(clock()-start_)/CLOCKS_PER_SEC<<"s"<<std::endl;  //输出时间（单位：ｓ）

        if(ros_visualization_ptr_ != nullptr && infer_param.vis_res){
            ros_visualization_ptr_->Visualization(*vGrid, frame_ptr->img_src_, frame_ptr->objs_);
        }
    }
}



};


