
#include "models/camera_model.h"

namespace waytous {
namespace deepinfer {
namespace camera {

bool CameraModel::Init(std::string& configPath) {
    // CUDA_CHECK(cudaStreamCreate(&stream_));
    std::string cfgPath = common::GetAbsolutePath(common::ConfigRoot::GetRootPath(), configPath);
    if(!common::PathExists(cfgPath)){
        LOG_WARN << "Init CameraModel config_file "<< common::ConfigRoot::GetRootPath() << " " << cfgPath << " not exist.";
        return false;
    }
    modelConfigNode = YAML::LoadFile(cfgPath);
    if (modelConfigNode.IsNull()) {
        LOG_WARN << "Init CameraModel, Load " << configPath << " failed! please check!";
        return false;
    }
    inputNames = modelConfigNode["inputNames"].as<std::vector<std::string>>();
    outputNames = modelConfigNode["outputNames"].as<std::vector<std::string>>();

    // units
    auto unitNodes = modelConfigNode["units"];
    for(int i=0; i<unitNodes.size(); i++){
        auto unitNode = unitNodes[i];
        std::string unitName = unitNode["name"].as<std::string>();
        LOG_INFO << "Init CameraModel unit: " << unitName; 
        interfaces::BaseUnitPtr unitPtr;
        unitPtr.reset(interfaces::BaseUnitRegisterer::GetInstanceByName(unitName));
        if(!unitPtr->Init(unitNode)){
            LOG_WARN << "Init CameraModel unit " << unitName << " failed!";
            return false;
        };
        units_.push_back(unitPtr);
    }
    return true;
}


bool CameraModel::Exec(std::vector<cv::Mat*> inputs, std::vector<interfaces::BaseIOPtr>& outputs){
    // set source
    for (int i=0; i<inputs.size(); i++){
        auto input_img = inputs[i];
        auto inputName = inputNames[i];
        LOG_INFO << "CameraModel input: " << inputName;
        auto input = std::make_shared<ios::CameraSrcIn>(ios::CameraSrcIn(input_img));
        interfaces::SetIOPtr(inputName, input);
    }
    // infer
    for(auto& unit: units_){
        LOG_INFO << "exec unit " << unit->Name();
        if(!unit->Exec()){
            LOG_ERROR << "exec CameraModel unit "<< unit->Name() << " exec error";
            return false;
        };
    }

    // output
    for(auto outName: outputNames){
        LOG_INFO << "CameraModel output: " << outName;
        outputs.push_back(interfaces::GetIOPtr(outName));
    }

    return true;
}


bool CameraModel::Exec(std::vector<base::Image8UPtr> inputs, std::vector<interfaces::BaseIOPtr>& outputs){
    // set source
    for (int i=0; i<inputs.size(); i++){
        auto input_img = inputs[i];
        auto inputName = inputNames[i];
        auto input = std::make_shared<ios::CameraSrcOut>(ios::CameraSrcOut(input_img));
        interfaces::SetIOPtr(inputName, input);
    }

    // infer
    for(auto& unit: units_){
        LOG_INFO << "exec unit" << unit->Name();
        if(!unit->Exec()){
            LOG_ERROR << "exec CameraModel unit "<< unit->Name() << " exec error";
            return false;
        };
    }

    // output
    for(auto outName: outputNames){
        outputs.push_back(interfaces::GetIOPtr(outName));
    }
    return true;
}


std::string CameraModel::Name(){
    return "CameraModel";
}



}  // namespace camera
}  // namespace deepinfer
}  // namespace waytous


