#include "libs/preprocessors/undistort.h"

namespace waytous {
namespace deepinfer {
namespace preprocess {


bool Undistort::Init(YAML::Node& node) {
    if(!BaseUnit::Init(node)){
        LOG_WARN << "Init trades postprocess error";
        return false;
    };

    width_ = node["imageWidth"].as<int>();
    height_ = node["imageHeight"].as<int>();
    std::string IntrinsicPath,  ExtrinsicPath = "";

    IntrinsicPath = node["IntrinsicPath"].as<std::string>();
    if(!loadIntrinsic(IntrinsicPath)){
        LOG_WARN << "Load intrinsic error: " << IntrinsicPath;
        return false;
    }

    d_mapx_.Reshape({height_, width_});
    d_mapy_.Reshape({height_, width_});

    Eigen::Matrix3f I;
    I << 1.f, 0.f, 0.f, 0.f, 1.f, 0.f, 0.f, 0.f, 1.f;

    InitUndistortRectifyMap(camera_intrinsic,
                            distortion_coefficients, I,
                            camera_intrinsic, width_,
                            height_, &d_mapx_, &d_mapy_);

    dst_img = std::make_shared<base::Image8U>(base::Image8U(height_, width_, base::Color::BGR));
    auto output = std::make_shared<ios::CameraSrcOut>(ios::CameraSrcOut(dst_img));
    interfaces::SetIOPtr(outputNames[0], output);
    inited_ = true;
    return true;
}


bool Undistort::loadIntrinsic(std::string& yaml_file){
    std::string yaml_path = common::GetAbsolutePath(common::ConfigRoot::GetRootPath(), yaml_file);
    if (!common::PathExists(yaml_path)) {
        LOG_INFO << yaml_path << " does not exist!";
        return false;
    }

    YAML::Node node = YAML::LoadFile(yaml_path);
    if (node.IsNull()) {
        LOG_INFO << "Load " << yaml_path << " failed! please check!";
        return false;
    }
    try{
        if(node["image_width"].IsDefined() && !node["image_width"].IsNull()){
            int width = node["image_width"].as<int>();
            if(width_ != width){
                LOG_INFO << "image width from cameraInfoConfig (" << width_<< ") and intrinsic (" << width << ") does not match, set with the intrinsic width. ";
                width_ = width;
            }
        }
        if(node["image_height"].IsDefined() && !node["image_height"].IsNull()){
            int height = node["image_height"].as<int>();
            if(height_ != height){
                LOG_INFO << "image height from cameraInfoConfig (" << height_<< ") and intrinsic (" << height << ") does not match, set with the intrinsic height. ";
                height_ = height;
            }
        }
        for(int i=0; i<9; i++){
            camera_intrinsic(i) = node["camera_matrix"]["data"][i].as<float>();
        }
        for(int i=0; i<5; i++){
            distortion_coefficients(i) = node["distortion_coefficients"]["data"][i].as<float>();
        }
        if(node["rectification_matrix"].IsDefined() && node["rectification_matrix"]["data"].IsDefined() &&
            !node["rectification_matrix"]["data"].IsNull()
        ){
            for(int i=0; i<9; i++){
                camera_rectification(i) = node["rectification_matrix"]["data"][i].as<float>();
            }
        }
        if(node["projection_matrix"].IsDefined() && node["projection_matrix"]["data"].IsDefined() &&
            !node["projection_matrix"]["data"].IsNull()
        ){
            for(int i=0; i<12; i++){
                camera_projection(i) = node["projection_matrix"]["data"][i].as<float>();
            }
        }
    } catch (YAML::Exception &e) {
        LOG_ERROR << "load camera params file " << yaml_path
            << " with error, YAML exception: " << e.what();
        return false;
    }
    return true;
}


void Undistort::InitUndistortRectifyMap(
    const Eigen::Matrix3f &camera_model,
    const Eigen::Matrix<float, 1, 5>& distortion, const Eigen::Matrix3f &R,
    const Eigen::Matrix3f &new_camera_model, int width, int height,
    base::Blob<float> *d_mapx, base::Blob<float> *d_mapy) {
    float fx = camera_model(0, 0);
    float fy = camera_model(1, 1);
    float cx = camera_model(0, 2);
    float cy = camera_model(1, 2);
    float nfx = new_camera_model(0, 0);
    float nfy = new_camera_model(1, 1);
    float ncx = new_camera_model(0, 2);
    float ncy = new_camera_model(1, 2);
    float k1 = distortion(0, 0);
    float k2 = distortion(0, 1);
    float p1 = distortion(0, 2);
    float p2 = distortion(0, 3);
    float k3 = distortion(0, 4);
    Eigen::Matrix3f Rinv = R.inverse();

    for (int v = 0; v < height_; ++v) {
        float *x_ptr = d_mapx->mutable_cpu_data() + d_mapx->offset(v);
        float *y_ptr = d_mapy->mutable_cpu_data() + d_mapy->offset(v);
        for (int u = 0; u < width_; ++u) {
        Eigen::Matrix<float, 3, 1> xy1;
        xy1 << (static_cast<float>(u) - ncx) / nfx,
            (static_cast<float>(v) - ncy) / nfy, 1;
        auto XYW = Rinv * xy1;
        double nx = XYW(0, 0) / XYW(2, 0);
        double ny = XYW(1, 0) / XYW(2, 0);
        double r_square = nx * nx + ny * ny;
        double scale = (1 + r_square * (k1 + r_square * (k2 + r_square * k3)));
        double nnx =
            nx * scale + 2 * p1 * nx * ny + p2 * (r_square + 2 * nx * nx);
        double nny =
            ny * scale + p1 * (r_square + 2 * ny * ny) + 2 * p2 * nx * ny;
        x_ptr[u] = static_cast<float>(nnx * fx + cx);
        y_ptr[u] = static_cast<float>(nny * fy + cy);
        }
    }
}


bool Undistort::Exec(){
    auto iptr = std::dynamic_pointer_cast<ios::CameraSrcOut>(interfaces::GetIOPtr(inputNames[0]));
    if (iptr == nullptr){
        LOG_ERROR << "Undistort input " << inputNames[0] << " haven't been init or doesn't exist.";
        return false;
    }
    if (!inited_) {
        LOG_WARN << "Undistortion not init.";
        return false;
    }

    auto src_img = iptr->img_ptr_;

    NppiInterpolationMode remap_mode = NPPI_INTER_LINEAR;
    NppiSize image_size;
    image_size.width = width_;
    image_size.height = height_;
    NppiRect remap_roi = {0, 0, width_, height_};

    NppStatus status;
    int d_map_step = static_cast<int>(d_mapx_.shape(1) * sizeof(float));
    switch (src_img->channels()) {
        case 1:
            status = nppiRemap_8u_C1R(
                src_img->gpu_data(), image_size, src_img->width_step(), remap_roi,
                d_mapx_.gpu_data(), d_map_step, d_mapy_.gpu_data(), d_map_step,
                dst_img->mutable_gpu_data(), dst_img->width_step(), image_size,
                remap_mode);
            break;
        case 3:
            status = nppiRemap_8u_C3R(
                src_img->gpu_data(), image_size, src_img->width_step(), remap_roi,
                d_mapx_.gpu_data(), d_map_step, d_mapy_.gpu_data(), d_map_step,
                dst_img->mutable_gpu_data(), dst_img->width_step(), image_size,
                remap_mode);
            break;
        default:
            LOG_ERROR << "Invalid number of channels: " << src_img->channels();
            return false;
    }

    if (status != NPP_SUCCESS) {
        LOG_ERROR << "NPP_CHECK_NPP - status = " << status;
        return false;
    }
    
    return true;
}


std::string Undistort::Name() {
    return "Undistort";
};


}  // namespace postprocess
}  // namespace deepinfer
}  // namespace waytous


