
#include "tasks/task_bsw.h"

namespace waytous{
namespace deepinfer{
namespace task{


bool TaskBSW::Init(std::string& taskConfigPath){
    if(!interfaces::BaseTask::Init(taskConfigPath)){
        LOG_ERROR << "Init task detect error";
        return false;
    };
    std::string detectorName = taskNode["detectorName"].as<std::string>();
    std::string detectorConfigPath = taskNode["detectorConfigPath"].as<std::string>();

    detector.reset(interfaces::BaseModelRegisterer::GetInstanceByName(detectorName));
    if(!detector->Init(detectorConfigPath)){
        LOG_ERROR << detectorName << " detector init problem";
        return false;
    };

    imageWidth = taskNode["imageWidth"].as<int>();
    imageHeight = taskNode["imageHeight"].as<int>();
    mask = cv::Mat::zeros(imageHeight, imageWidth, CV_8UC3);

    // init distance pools
    std::string warningDistancePoolsPath = taskNode["warningDistancePoolsPath"].as<std::string>();
    std::string distanceConfigPath = common::GetAbsolutePath(common::ConfigRoot::GetRootPath(), warningDistancePoolsPath);
    if(!common::PathExists(distanceConfigPath)){
        LOG_ERROR << "camera distance pool config_file " << distanceConfigPath << " not exist.";
        return false;
    }
    YAML::Node distanceNode = YAML::LoadFile(distanceConfigPath);
    if (distanceNode.IsNull()) {
        LOG_ERROR << "Load " << distanceConfigPath << " failed! please check!";
        return false;
    }
    
    // init ranges
    std::string warningRangeConfigPath = taskNode["warningRangeConfigPath"].as<std::string>();
    if(!InitRanges(warningRangeConfigPath, distanceNode)){
        LOG_ERROR << "Init BSW warning range error.";
        return false;
    }
    return true;
}


bool TaskBSW::Exec(std::vector<cv::Mat*> inputs, std::vector<interfaces::BaseIOPtr>& outputs){
    if(inputs.size() != 1){
        LOG_ERROR << 'Now only support infer one image once.';
        return false;
    }
    std::vector<interfaces::BaseIOPtr> dets;
    if(!detector->Exec(inputs, dets)){
        LOG_ERROR << "Task Detect Exec error";
        return false;
    };

    auto objs = std::dynamic_pointer_cast<ios::Detection2Ds>(dets[0])->detections;
    // delete lost tracklets
    for(auto& bo_pair: BSWRes){
        auto bo = bo_pair.second;
        bool matched = false;
        for(auto& obj: objs){
            if(bo->obj->track_id == obj->track_id){
                matched = true;
                break;
            }
        }
        if(!matched){
            bo->lost_ages += 1;
            if(bo->lost_ages > 10){
                BSWRes.erase(bo_pair.first);
            }
        }
    }

    for(auto &obj : objs){
        auto k = BSWRes.find(obj->track_id);
        if(k == BSWRes.end()){
            BSWRes[obj->track_id] = std::make_shared<BSWResult>(BSWResult());
            BSWRes[obj->track_id]->counters = std::vector<int>(ranges.size(), 0);
        }
        auto& bo = BSWRes[obj->track_id];
        bo->obj = obj;
        // if(obj->class_name != "person") continue;
        float buttom_cx = (obj->x1 + obj->x2) / 2;
        float buttom_cy = obj->y2;
        bool isIn = false;
        for(int i = ranges.size() - 1; i >=0; i--){ // start from the worst warning.
            if(isIn){
                bo->counters[i] += 1;
                continue;
            }
            isIn = isInRange(ranges[i].polygan, buttom_cx, buttom_cy);
            if(isIn){
                bo->counters[i] += 1;
                if(bo->counters[i] > ranges[i].warningFrameThreshold){
                    bo->msg = WarningName2Msg[ranges[i].name];
                }
            }else{
                bo->counters[i] = 0;
            }
        }
        outputs.push_back(bo);
    }
    return true;
}


void TaskBSW::Visualize(cv::Mat* image, interfaces::BaseIOPtr outs){
    for(auto &bo_pair : BSWRes){
        auto& bo = bo_pair.second;
        if(bo->lost_ages > 0){
            continue;
        }
        auto &obj = bo->obj;
        cv::Scalar color = bo->counters[0] % 4 == 0 ? cv::Scalar(255, 255, 255) : ranges[(int) bo->msg].color;
        cv::rectangle(*image, cv::Rect(int(obj->x1), int(obj->y1), int(obj->x2 - obj->x1), int(obj->x2 - obj->y1)), color, 2);
        cv::putText(*image, std::to_string(obj->track_id) + "." + obj->class_name + ":" + common::formatValue(obj->confidence, 2), 
                cv::Point(int(obj->x1), int(obj->y1) - 5), cv::FONT_HERSHEY_SIMPLEX, 0.5, color, 1);
    }
    // draw outlines
    cv::addWeighted(*image, 0.9, mask, 0.1, 0, *image);

    // // draw warning
    // cv::putText(image, r.name, cv::Point(100, 100), cv::FONT_HERSHEY_SIMPLEX, 1, r.color, 1);
}


bool TaskBSW::InitRanges(const std::string& warningRangeConfigPath, YAML::Node& distanceNode){
    std::string configPath = common::GetAbsolutePath(common::ConfigRoot::GetRootPath(), warningRangeConfigPath);
    if(!common::PathExists(configPath)){
        LOG_ERROR << "blind area config_file " << configPath << " not exist.";
        return false;
    }
    YAML::Node rangeNode = YAML::LoadFile(configPath);
    if (rangeNode.IsNull()) {
        LOG_ERROR << "Load " << configPath << " failed! please check!";
        return false;
    }
    bool useFullPolygan = taskNode["useFullPolygan"].as<bool>();
    auto infoNode = rangeNode["warningInfos"];
    for(int i=0; i < infoNode.size(); i++){
        RangeInfo rinfo;
        auto curNode = infoNode[i];
        rinfo.name = curNode["name"].as<std::string>();
        auto color = curNode["color"].as<std::vector<int>>();
        rinfo.color = cv::Scalar(color[0], color[1], color[2]);
        rinfo.warningFrameThreshold = curNode["warningFrameThreshold"].as<int>();

        float distance = curNode["distance"].as<float>();
        std::vector<int> point;
        if(distance > 0){
            std::string distanceNameInNode = "dist_" + std::to_string(int(distance * 100)); // meter->centermeter
            auto polygonNode = distanceNode[distanceNameInNode];
            if(!polygonNode.IsDefined()){
                LOG_ERROR << rinfo.name << ", camera distance " << distance << " m, cannot find in pool.";
                return false;
            }
            // LOG_INFO << rinfo.name << " " << rinfo.warningFrameThreshold;
            for(int j=0; j < polygonNode.size(); j++){
                point = polygonNode[j].as<std::vector<int>>();
                if(!useFullPolygan && j == 0){
                    if(point[0] < imageHeight - point[1]){
                        rinfo.polygan.push_back(cv::Point(0, imageHeight));// fisrt left-buttom
                        rinfo.polygan.push_back(cv::Point(0, point[1])); // second left-top
                    }else{
                        rinfo.polygan.push_back(cv::Point(point[0], imageHeight)); // only left-buttom
                    }
                }
                if(imageWidth - point[0] <= 1){ // 1919->1920
                    point[0] = imageWidth;
                }
                if(imageHeight - point[1] <= 1){ // 1079->1080
                    point[1] = imageHeight;
                }
                rinfo.polygan.push_back(cv::Point(point[0], point[1]));
            }
            if(!useFullPolygan){
                if(imageWidth - point[0] < imageHeight - point[1]){
                    rinfo.polygan.push_back(cv::Point(imageWidth, point[1]));
                    rinfo.polygan.push_back(cv::Point(imageWidth, imageHeight));
                }else{
                    rinfo.polygan.push_back(cv::Point(point[0], imageHeight));
                }
            }
        }else{
            // distance < 0, if means distance = inf, set polygan with the image's outline.
            rinfo.polygan.push_back(cv::Point(0, imageHeight));
            rinfo.polygan.push_back(cv::Point(0, 0));
            rinfo.polygan.push_back(cv::Point(imageWidth, 0));
            rinfo.polygan.push_back(cv::Point(imageWidth, imageHeight));
        }

        cv::fillPoly(mask, rinfo.polygan, rinfo.color);
        ranges.push_back(rinfo);
    }
    return true;
};


bool TaskBSW::isInRange(const std::vector<cv::Point>& polygan, float x, float y){
    bool isIn = false;
    int j = polygan.size() - 1;
    for(int i=0; i < polygan.size(); i++ ){
        auto& pi = polygan[i];
        auto& pj = polygan[j];
        if((pi.y < y && pj.y >= y || pj.y < y && pi.y >= y) && 
            (pi.x <= x || pj.x <= x)){
                isIn ^= (pi.x + (y - pi.y) / (pj.y - pi.y + 1e-5) * (pj.x - pi.x) < x);
        }
        j = i;
    }
    return isIn;
}


std::string TaskBSW::Name(){
    return "TaskBSW";
}


} //namespace task
} //namspace deepinfer
} //namespace waytous









