#ifndef DEEPINFER_BASE_PIPELINE_H_
#define DEEPINFER_BASE_PIPELINE_H_

#include <algorithm>
#include <iostream>
#include <numeric>
#include <string>
#include <unordered_set>
#include <utility>
#include <vector>
#include <thread>
#include <atomic>
#include <yaml-cpp/yaml.h>

#include "common/register.h"

using namespace std;

namespace waytous {
namespace deepinfer {
namespace base {

class LifeCycle {
public:
    virtual void init(YAML::Node& node) = 0;

    virtual void startUp() = 0;

    virtual void shutDown() = 0;
};

template<typename T>
class Component : public LifeCycle {
public:
    virtual std::string getName() = 0;

    virtual void execute(T t) = 0;
};

template<typename T, typename R>
class AbstractComponent : public Component<T> {
private:
    std::unordered_set<shared_ptr<Component<R>>> down_stream;
protected:
    const std::unordered_set<shared_ptr<Component<R>>> &getDownStream() {
        return down_stream;
    }

protected:
    virtual R doExecute(T t) = 0;

public:
    void addDownStream(shared_ptr<Component<R>> component) {
        down_stream.insert(component);
    }

    void init(YAML::Node& node) override {

    }

    void execute(T t) override {
        R r = doExecute(t);
        cout << this->getName() + "\treceive\t" << typeid(t).name() << "\t" << t << "\treturn\t" << typeid(r).name()
             << "\t" << r << endl;
        if constexpr (is_same_v<R, void>) {
            return;
        }
        for (auto &&obj : getDownStream()) {
            obj->execute(r);
        }
    }

    void startUp() override {
        for (auto &&obj : this->getDownStream()) {
            obj->startUp();
        }
        cout << "------------------ " + this->getName() + " is starting ----------------------" << endl;
    }

    void shutDown() override {
        auto downStreams = this->getDownStream();
        for (auto &&obj : downStreams) {
            obj->shutDown();
        }
        cout << "------------------ " + this->getName() + " is ending ----------------------" << endl;
    }


};

template<typename T, typename R>
using Source = AbstractComponent<T, R>;
// DEEPINFER_REGISTER_REGISTERER(Source);
// #define DEEPINFER_REGISTER_INFERENCE_ENGINE(name) \
//   DEEPINFER_REGISTER_CLASS(Source, name)

template<typename T, typename R>
using Channel = AbstractComponent<T, R>;

template<typename T, typename R>
using Sink = AbstractComponent<T, R>;


template<typename R, typename T>
class Pipeline : public LifeCycle {
private:
    shared_ptr<Source<R, T>> source;

public:
    void setSource(shared_ptr<Source<R, T>> component) {
        source = component;
    }

    void init(YAML::Node& node) override {
    }

    void startUp() override {
        assert(source.get() != nullptr);
        source->startUp();
    }

    void shutDown() override {
        source->shutDown();
    }
};

/*
class printSink;

class intStringChannel;

class printSink : public Sink<string, int> {
 public:
    string getName() override {
        return "printSink";
    }

 protected:
    int doExecute(string t) override {
        return INT_MIN;
    }
};

class intStringChannel : public Channel<int, string> {
 public:
    void init(std::string config) override {

    }

    string getName() override {
        return "intStringChannel";
    }

    void startUp() override {

    }

    void shutDown() override {

    }

 protected:
    string doExecute(int t) override {
        return to_string(t + 100);
    }
};

class IntSource : public Source<int, int> {
 private:
    int val = 0;
 public:
    void init(std::string config) override {
        cout << "--------- " + getName() + " init --------- ";
        val = 1;
    }

    string getName() override {
        return "Int Source";
    }

    void startUp() override {
        this->execute(val);
    }

 protected:
    int doExecute(int) override {
        return val + 1;
    }
};


int main() {
    pipeline<int, int> p;
    // source
    auto is = make_shared<IntSource>();

    // channel
    auto isc = make_shared<intStringChannel>();

    // sink
    auto ps = make_shared<printSink>();

    is->addDownStream(isc);
    isc->addDownStream(ps);

    // 设置 source
    p.setSource(is);

    // 启动
    p.startUp();
}
*/


}  // namespace base
}  // namespace deepinfer
}  // namespace waytous

#endif

