package com.waytous.pushassistant

import android.media.CamcorderProfile
import android.os.Environment
import com.blankj.utilcode.util.FileUtils
import com.blankj.utilcode.util.LogUtils
import com.blankj.utilcode.util.SDCardUtils
import com.mediatek.smartplatform.RecordConfiguration
import com.mediatek.smartplatform.RecordConfiguration.VideoFrameMode.VIDEO_FRAME_MODE_DUAL_SOURCE
import com.mediatek.smartplatform.RecordSource
import com.mediatek.smartplatform.SpmCameraDevice
import com.mediatek.smartplatform.SpmCameraDevice.VideoCallback
import com.mediatek.smartplatform.VideoInfoMap
import kotlinx.coroutines.cancel
import java.io.File

class DmsRecordManager(private val camera: SpmCameraDevice, private val recordParent: File, private val duration:Int, private val frameRate:Int = 15, private val bitRate:Int = 90000) {

    private val dmsDir = if (SDCardUtils.getSDCardInfo().size > 1){
        File(SDCardUtils.getSDCardInfo()[1].path, "waytous/alarm/dms")
    } else {
        File(Environment.getExternalStorageDirectory(), "waytous/alarm/dms")
    }

    private val alarmVideoPath = File(dmsDir, "video").path.also {
        FileUtils.createOrExistsFile(it)
    }

    private val alarmPicturePath = File(dmsDir, "picture").path.also {
        FileUtils.createOrExistsFile(it)
    }


    fun startRecord() {
        val dmsPlaybackDir = File(recordParent, "dms")
        if (!dmsPlaybackDir.exists()) {
            dmsPlaybackDir.mkdirs()
        }
        try {
            camera.startRecord(RecordSource.GENERAL_CAMERA, createRecordConfiguration(dmsPlaybackDir.path,duration,frameRate,bitRate))
        } catch (e: Exception) {
            camera.stopRecord(RecordSource.GENERAL_CAMERA)
            LogUtils.iTag("camera${camera.cameraId}", "start record error: ${e.message}")
            LogUtils.file("camera${camera.cameraId}", "start record error: ${e.message}")
        }
    }

    fun stopRecord(){
        camera.stopRecord(RecordSource.GENERAL_CAMERA)
    }


    private fun createRecordConfiguration(path:String, duration:Int, frameRate:Int, bitRate:Int): RecordConfiguration = RecordConfiguration.get(
        RecordSource.GENERAL_CAMERA).apply {
        mCamcorderProfile = CamcorderProfile.get(CamcorderProfile.QUALITY_480P).apply {
            videoFrameRate = frameRate
            videoBitRate = bitRate
        }
        mOutPutFilePath = path
        mOutPutFileName = "playback-%Y%m%d_%H%M%S"
        mVideoCycleDeleteFileNum = 30
        mVideoRotateDuration = duration
        mLockFileNamePrefix = "alarm-"
        mLockFilePath = alarmVideoPath
        mVideoFrameMode = VIDEO_FRAME_MODE_DUAL_SOURCE
        mVideoCallback = object: VideoCallback {
            override fun onVideoTaken(videoInfoMap: VideoInfoMap?) {
                when (videoInfoMap?.getInt(VideoInfoMap.KEY_RECORDER_EVENT_TYPE)) {
                    VideoCallback.VIDEO_EVENT_SDCARD_FULL -> {
                        LogUtils.eTag("camera0","Storage Full")
                        LogUtils.file("camera0","Storage Full")
                    }

                    VideoCallback.VIDEO_EVENT_RECORD_SDCARD_DAMAGED -> {
                        LogUtils.eTag("camera0","sdcard damage")
                        LogUtils.file("camera0","sdcard damage")
                    }

                    VideoCallback.VIDEO_EVENT_RECORD_RECORDING_ERROR -> {
                        LogUtils.eTag("camera0","error duration recording")
                        LogUtils.file("camera0","error duration recording")
                    }

                    VideoCallback.VIDEO_EVENT_ADD_FILE_IN_GALLERY -> {

                    }

                    VideoCallback.VIDEO_EVENT_KEYPOINT_STOP -> {
                        val videoPath = videoInfoMap.getString(VideoInfoMap.KEY_RECORDER_FILE_PATH)
                        LogUtils.iTag("dms","lock video complete,$videoPath")
                        LogUtils.file("dms","lock video complete,$videoPath")
                    }
                }
            }

            override fun onVideoFrame(h264: ByteArray, dataType: Int, size: Int, cameraId: String?, recordType: Int) {
                PushTools.onCameraDmsFrame(h264,size)
            }
        }
    }

    companion object {

        const val TOPIC_DMS_ALARM = "/alarm"

        const val MOD_DMS_CALL = 0x02

        const val MOD_DMS_SMOKE = 0x03

        const val MOD_DMS_YAWN = 0x01

        const val MOD_DMS_DISTRACT = 0x04

        const val MOD_DMS_DRIVER_ABNORMAL = 0x05

        const val MOD_DMS_LENS_COVERED = 0x06

        const val MOD_DMS_SEATBELT_UNFASTENED = 0x07

        const val MOD_DMS_CLOSE_EYE = 0x08

    }
}