package com.waytous.pushassistant

import android.app.Service
import android.content.Intent
import android.os.Binder
import android.os.Environment
import android.os.Handler
import android.os.IBinder
import android.os.Looper
import android.os.Message
import com.blankj.utilcode.util.LogUtils
import com.blankj.utilcode.util.SDCardUtils
import com.mediatek.smartplatform.SmartPlatformManager
import com.mediatek.smartplatform.SpmCameraDevice
import com.mediatek.smartplatform.SpmCameraDevice.ErrorCallback
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancel
import java.io.File
import java.lang.ref.WeakReference

class CameraMonitor : Service(), ErrorCallback {

    override fun onBind(intent: Intent): IBinder = CameraMonitorBinder()

    private lateinit var platformManager: SmartPlatformManager

    private val monitorHandler = CameraMonitorHandler(this)

    private var _dmsCamera: SpmCameraDevice? = null

    private var _frontCamera: SpmCameraDevice? = null

    private var _backCamera: SpmCameraDevice? = null

    private var _leftCamera: SpmCameraDevice? = null

    private var _rightCamera: SpmCameraDevice? = null

    private var _bsdCamera: SpmCameraDevice? = null

    private var jt0178Job: Job? = null

    private var dmsRecordManager: DmsRecordManager? = null

    private var frontRecordManager: FrontRecordManager? = null

    private var rightRecordManager: RightRecordManager? = null

    private var backRecordManager: BackRecordManager? = null

    private var leftRecordManager: LeftRecordManager? = null

    private var bsdRecordManager: BsdRecordManager? = null

    private val playbackDir = if (SDCardUtils.getSDCardInfo().size > 1) File(SDCardUtils.getSDCardInfo()[1].path, "waytous/playback")
    else File(Environment.getExternalStorageDirectory(), "waytous/playback")

    override fun onCreate() {
        super.onCreate()
        platformManager = SmartPlatformManager.get()
        if (!platformManager.isServiceAlive) {
            monitorHandler.sendEmptyMessage(MSG_SMART_DEVICE_SERVICE_UNINITIALIZED)
        } else {
            openCameraDevices(playbackDir)
        }
    }

    override fun onError(errorCode: Int, cameraId: String?, device: SpmCameraDevice?) {
        val message = Message.obtain()
        message.arg1 = errorCode
        message.obj = cameraId
        message.what = MSG_DEVICE_ERROR_CALLBACK
        monitorHandler.sendMessageDelayed(message,50)
        LogUtils.dTag(TAG, "open camera $cameraId error,$errorCode")
        LogUtils.file(TAG, "open camera $cameraId error,$errorCode")
    }

    override fun onDestroy() {
        super.onDestroy()
        jt0178Job?.cancel("CameraMonitor stopped")
        dmsRecordManager?.stopRecord()
        frontRecordManager?.stopRecord()
        rightRecordManager?.stopRecord()
        backRecordManager?.stopRecord()
        leftRecordManager?.stopRecord()
        bsdRecordManager?.stopRecord()
    }

    private fun openCameraDevices(playbackDir: File) {
        try {
            _dmsCamera = platformManager.openCameraDevice(CAMERA_DMS_DEV).apply {
                setErrorCallback(this@CameraMonitor)
            }
            dmsRecordManager = DmsRecordManager(_dmsCamera!!, playbackDir, 5 * 60 * 1000, 15, 90000)
            dmsRecordManager?.startRecord()
        } catch (e: Exception) {
            dmsRecordManager?.stopRecord()
            _dmsCamera = null
            dmsRecordManager = null
            LogUtils.eTag(TAG, "open camera error:${e.message},camera id:$CAMERA_DMS_DEV")
            LogUtils.file(TAG, "open camera error:${e.message},camera id:$CAMERA_DMS_DEV")
        }
        try {
            _frontCamera = platformManager.openCameraDevice(CAMERA_FRONT_DEV).apply {
                setErrorCallback(this@CameraMonitor)
            }
            frontRecordManager = FrontRecordManager(_frontCamera!!, playbackDir, 5 * 60 * 1000, 15, 90000)
            frontRecordManager?.startRecord()
        } catch (e: Exception) {
            frontRecordManager?.stopRecord()
            _frontCamera = null
            frontRecordManager = null
            LogUtils.eTag(TAG, "open camera error:${e.message},camera id:$CAMERA_FRONT_DEV")
            LogUtils.file(TAG, "open camera error:${e.message},camera id:$CAMERA_FRONT_DEV")
        }
        try {
            _backCamera = platformManager.openCameraDevice(CAMERA_BACK_DEV).apply {
                setErrorCallback(this@CameraMonitor)
            }
            backRecordManager = BackRecordManager(_backCamera!!, playbackDir, 5 * 60 * 1000, 15, 90000)
            backRecordManager?.startRecord()
        } catch (e: Exception) {
            backRecordManager?.stopRecord()
            _backCamera = null
            backRecordManager = null
            LogUtils.eTag(TAG, "open camera error:${e.message},camera id:$CAMERA_BACK_DEV")
            LogUtils.file(TAG, "open camera error:${e.message},camera id:$CAMERA_BACK_DEV")
        }
        try {
            _leftCamera = platformManager.openCameraDevice(CAMERA_LEFT_DEV).apply {
                setErrorCallback(this@CameraMonitor)
            }
            leftRecordManager = LeftRecordManager(_leftCamera!!, playbackDir, 5 * 60 * 1000, 15, 90000)
            leftRecordManager?.startRecord()
        } catch (e: Exception) {
            leftRecordManager?.stopRecord()
            _leftCamera = null
            leftRecordManager = null
            LogUtils.eTag(TAG, "open camera error:${e.message},camera id:$CAMERA_LEFT_DEV")
            LogUtils.file(TAG, "open camera error:${e.message},camera id:$CAMERA_LEFT_DEV")
        }
        try {
            _rightCamera = platformManager.openCameraDevice(CAMERA_RIGHT_DEV).apply {
                setErrorCallback(this@CameraMonitor)
            }
            rightRecordManager = RightRecordManager(_rightCamera!!, playbackDir, 5 * 60 * 1000, 15, 90000)
            rightRecordManager?.startRecord()
        } catch (e: Exception) {
            rightRecordManager?.stopRecord()
            _rightCamera = null
            rightRecordManager = null
            LogUtils.eTag(TAG, "open camera error:${e.message},camera id:$CAMERA_RIGHT_DEV")
            LogUtils.file(TAG, "open camera error:${e.message},camera id:$CAMERA_RIGHT_DEV")
        }
        try {
            _bsdCamera = platformManager.openCameraDevice(CAMERA_BSD_DEV).apply {
                setErrorCallback(this@CameraMonitor)
            }
            bsdRecordManager = BsdRecordManager(_bsdCamera!!, playbackDir, 5 * 60 * 1000, 15, 90000)
            bsdRecordManager?.startRecord()
        } catch (e: Exception) {
            bsdRecordManager?.stopRecord()
            _bsdCamera = null
            bsdRecordManager = null
            LogUtils.eTag(TAG, "open camera error:${e.message},camera id:$CAMERA_BSD_DEV")
            LogUtils.file(TAG, "open camera error:${e.message},camera id:$CAMERA_BSD_DEV")
        }
    }

    fun getDmsCamera(): SpmCameraDevice? = _dmsCamera

    fun getFrontCamera(): SpmCameraDevice? = _frontCamera

    fun getBackCamera(): SpmCameraDevice? = _backCamera

    fun getLeftCamera(): SpmCameraDevice? = _leftCamera

    fun getRightCamera(): SpmCameraDevice? = _rightCamera

    fun getBsdCamera(): SpmCameraDevice? = _bsdCamera

    inner class CameraMonitorBinder : Binder() {
        fun getService(): CameraMonitor = this@CameraMonitor
    }

    class CameraMonitorHandler(cameraMonitor: CameraMonitor) : Handler(Looper.getMainLooper()) {

        private val monitor = WeakReference(cameraMonitor)

        override fun handleMessage(msg: Message) {
            LogUtils.iTag(TAG, "camera monitor handle message:${msg}")
            LogUtils.file(TAG, "camera monitor handle message:${msg}")
            when (msg.what) {
                MSG_SMART_DEVICE_SERVICE_UNINITIALIZED -> {
                    monitor.get()?.platformManager = SmartPlatformManager.get()
                    monitor.get()?.openCameraDevices(monitor.get()?.playbackDir!!)
                }
                MSG_DEVICE_ERROR_CALLBACK -> {
                    when (val cameraId = msg.obj.toString()) {
                        CAMERA_DMS_DEV -> {
                            monitor.get()?.dmsRecordManager?.stopRecord()
                            monitor.get()?.platformManager?.closeCameraDevice(cameraId)
                            monitor.get()?._dmsCamera = null
                            monitor.get()?.dmsRecordManager = null
                            monitor.get()?._dmsCamera = monitor.get()?.platformManager?.openCameraDevice(cameraId)?.apply {
                                setErrorCallback(monitor.get())
                            }
                            monitor.get()?._dmsCamera?.let {
                                monitor.get()?.dmsRecordManager = DmsRecordManager(it, monitor.get()?.playbackDir!!, 5 * 60 * 1000, 15, 90000)
                                monitor.get()?.dmsRecordManager?.startRecord()
                            }
                        }
                        CAMERA_FRONT_DEV -> {
                            monitor.get()?.frontRecordManager?.stopRecord()
                            monitor.get()?.platformManager?.closeCameraDevice(cameraId)
                            monitor.get()?._frontCamera = null
                            monitor.get()?.frontRecordManager = null
                            monitor.get()?._frontCamera = monitor.get()?.platformManager?.openCameraDevice(cameraId)?.apply {
                                setErrorCallback(monitor.get())
                            }
                            monitor.get()?._frontCamera?.let {
                                monitor.get()?.frontRecordManager = FrontRecordManager(it, monitor.get()?.playbackDir!!, 5 * 60 * 1000, 15, 90000)
                                monitor.get()?.frontRecordManager?.startRecord()
                            }
                        }

                        CAMERA_BACK_DEV -> {
                            monitor.get()?.backRecordManager?.stopRecord()
                            monitor.get()?.platformManager?.closeCameraDevice(cameraId)
                            monitor.get()?._backCamera = null
                            monitor.get()?.backRecordManager = null
                            monitor.get()?._backCamera = monitor.get()?.platformManager?.openCameraDevice(cameraId)?.apply {
                                setErrorCallback(monitor.get())
                            }
                            monitor.get()?._backCamera?.let {
                                monitor.get()?.backRecordManager = BackRecordManager(it,monitor.get()?.playbackDir!!, 5 * 60 * 1000, 15, 90000)
                                monitor.get()?.backRecordManager?.startRecord()
                            }
                        }

                        CAMERA_LEFT_DEV -> {
                            monitor.get()?.leftRecordManager?.stopRecord()
                            monitor.get()?.platformManager?.closeCameraDevice(cameraId)
                            monitor.get()?._leftCamera = null
                            monitor.get()?.leftRecordManager = null
                            monitor.get()?._leftCamera = monitor.get()?.platformManager?.openCameraDevice(cameraId)?.apply {
                                setErrorCallback(monitor.get())
                            }
                            monitor.get()?._leftCamera?.let {
                                monitor.get()?.leftRecordManager = LeftRecordManager(it, monitor.get()?.playbackDir!!, 5 * 60 * 1000, 15, 90000)
                                monitor.get()?.leftRecordManager?.startRecord()
                            }
                        }

                        CAMERA_RIGHT_DEV -> {
                            monitor.get()?.rightRecordManager?.stopRecord()
                            monitor.get()?.platformManager?.closeCameraDevice(cameraId)
                            monitor.get()?._rightCamera = null
                            monitor.get()?.rightRecordManager = null
                            monitor.get()?._rightCamera = monitor.get()?.platformManager?.openCameraDevice(cameraId)?.apply {
                                setErrorCallback(monitor.get())
                            }
                            monitor.get()?._rightCamera?.let {
                                monitor.get()?.rightRecordManager = RightRecordManager(it, monitor.get()?.playbackDir!!, 5 * 60 * 1000, 15, 90000)
                                monitor.get()?.rightRecordManager?.startRecord()
                            }
                        }

                        CAMERA_BSD_DEV -> {
                            monitor.get()?.bsdRecordManager?.stopRecord()
                            monitor.get()?.platformManager?.closeCameraDevice(cameraId)
                            monitor.get()?._bsdCamera= null
                            monitor.get()?.bsdRecordManager = null
                            monitor.get()?._bsdCamera = monitor.get()?.platformManager?.openCameraDevice(cameraId)?.apply {
                                setErrorCallback(monitor.get())
                            }
                            monitor.get()?._bsdCamera?.let {
                                monitor.get()?.bsdRecordManager = BsdRecordManager(it, monitor.get()?.playbackDir!!, 5 * 60 * 1000, 15, 90000)
                                monitor.get()?.bsdRecordManager?.startRecord()
                            }
                        }
                    }
                }
            }

        }
    }

    companion object {

        private const val TAG = "CameraMonitor"

        const val MSG_DEVICE_ERROR_CALLBACK = 34

        const val CAMERA_DMS_DEV = "0"

        const val CAMERA_FRONT_DEV = "1"

        const val CAMERA_BACK_DEV = "2"

        const val CAMERA_LEFT_DEV = "3"

        const val CAMERA_RIGHT_DEV = "4"

        const val CAMERA_BSD_DEV = "5"

        const val MSG_SMART_DEVICE_SERVICE_UNINITIALIZED = 31
    }
}