import type { RouteObject } from 'react-router';
import { Outlet, useLocation, useNavigate } from 'react-router';
import { useEffect } from 'react';

import Register from '@/pages/register';
import Tasks from '@/pages/tasks';
import TaskEdit from '@/pages/tasks.[id].edit';
import TaskAnnotation from '@/pages/tasks.[id].samples.[id]';
import Samples from '@/pages/tasks.[id]';
import TaskSamplesFinished from '@/pages/tasks.[id].samples.finished';
import Page404 from '@/pages/404';
import MainLayout from '@/layouts/MainLayoutWithNavigation';

import type { TaskLoaderResult } from './loaders/task.loader';
import { taskLoader, tasksLoader } from './loaders/task.loader';
import { rootLoader } from './loaders/root.loader';
import { sampleLoader } from './loaders/sample.loader';
import RequireAuth from './components/RequireSSO';
import { registerLoader } from './loaders/register.loader';
import { loginLoader } from './loaders/login.loader';
import LoginPage from './pages/login';
import ClassifyPage from './pages/classify';
import GestureCollectionPage from './pages/gesture/collection';
import GestureValidationPage from './pages/gesture/validation';
import GestureSettingPage from './pages/gesture/setting';





import FaceCollectionPage from './pages/face/collection';
import FaceValidationPage from './pages/face/validation';
import FaceSettingPage from './pages/face/setting';

import VoicePage from './pages/voice/voice';
import VoiceCollectionPage from './pages/voice/voiceCollection';
import VoiceSettingPage from './pages/voice/setting';

import ScoringPage from './pages/scoring';


import UIUEConsolePage from './pages/UIUE/Console';
import UIUEUserControlPage from './pages/UIUE/UserControl';



function Root() {
  const location = useLocation();
  const navigate = useNavigate();

  useEffect(() => {
    // 如果是根路径，跳转到任务管理（以任务管理为首页）
    if (location.pathname === '/' || location.pathname === '') {
      navigate('/classify');
    }
  }, [location.pathname, navigate]);

  if (!window.IS_ONLINE) {
    return <Outlet />;
  }

  return (
    <RequireAuth>
      <Outlet />
    </RequireAuth>
  );
}

const routes: RouteObject[] = [
  {
    path: '/',
    id: 'root',
    element: <Root />,
    loader: rootLoader,
    children: [
      {
        path: 'tasks',
        element: <MainLayout />,
        errorElement: <Page404 />,
        id: 'tasks',
        loader: tasksLoader,
        handle: {
          crumb: () => {
            return '任务列表';
          },
        },
        children: [
          {
            index: true,
            element: <Tasks />,
          },
          {
            path: ':taskId',
            id: 'task',
            element: <Outlet />,
            loader: taskLoader,
            handle: {
              crumb: (data: TaskLoaderResult) => {
                return data?.task?.name;
              },
            },
            children: [
              {
                index: true,
                element: <Samples />,
              },
              {
                path: 'edit',
                element: <TaskEdit />,
                loader: async ({ params }) => {
                  return params.taskId !== '0' ? '任务编辑' : '新建任务';
                },
                handle: {
                  crumb: (data: string) => {
                    return data;
                  },
                },
              },
              {
                path: 'samples',
                id: 'samples',
                element: <Outlet />,
                children: [
                  {
                    path: ':sampleId',
                    element: <TaskAnnotation />,
                    loader: sampleLoader,
                    id: 'annotation',
                    handle: {
                      crumb: () => {
                        return '开始标注';
                      },
                    },
                  },
                  {
                    path: 'finished',
                    element: <TaskSamplesFinished />,
                    loader: taskLoader,
                    handle: {
                      crumb: () => {
                        return '标注结束';
                      },
                    },
                  },
                ],
              },
            ],
          },
        ],
      },
      {
        path: 'scoring',
        element: <MainLayout />,
        errorElement: <Page404 />,
        id: 'scoring',
        loader: tasksLoader,
        handle: {
          crumb: () => {
            return '标注结果';
          },
        },
        children: [
          {
            index: true,
            element: <ScoringPage />,
          },
        ],
      },
      {
        path: 'classify',
        element: <MainLayout />,
        errorElement: <Page404 />,
        id: 'classify',
        loader: tasksLoader,
        handle: {
          crumb: () => {
            return '功能选择';
          },
        },
        children: [
          {
            index: true,
            element: <ClassifyPage />,
          },
          {
            path: 'voice',
            element: <Outlet />,
            errorElement: <Page404 />,
            id: 'voice',
            loader: tasksLoader,
            handle: {
              crumb: () => {
                return '语音模型体验';
              },
            },
            children: [
              {
                index: true,
                element: <VoicePage />,
              },
            ],
          },
          {
            path: 'voiceCollection',
            element: <Outlet />,
            errorElement: <Page404 />,
            id: 'voiceCollection',
            loader: tasksLoader,
            handle: {
              crumb: () => {
                return '语音采集';
              },
            },
            children: [
              {
                index: true,
                element: <VoiceCollectionPage />,
              },
            ],
          },
          {
            path: 'voicesetting',
            element: <Outlet />,
            errorElement: <Page404 />,
            id: 'voicesetting',
            loader: tasksLoader,
            handle: {
              crumb: () => {
                return '语音训练参数设置';
              },
            },
            children: [
              {
                index: true,
                element: <VoiceSettingPage />,
              },
            ],
          },
          {
            path: 'gesturesetting',
            element: <Outlet />,
            errorElement: <Page404 />,
            id: 'gesturesetting',
            loader: tasksLoader,
            handle: {
              crumb: () => {
                return '手势训练参数设置';
              },
            },
            children: [
              {
                index: true,
                element: <GestureSettingPage />,
              },
            ],
          },
          {
            path: 'gesturecollection',
            element: <Outlet />,
            errorElement: <Page404 />,
            id: 'gesturecollection',
            loader: tasksLoader,
            handle: {
              crumb: () => {
                return '手势视频采集';
              },
            },
            children: [
              {
                index: true,
                element: <GestureCollectionPage />,
              },
            ],
          },
          {
            path: 'gesturevalidation',
            element: <Outlet />,
            errorElement: <Page404 />,
            id: 'gesturevalidation',
            loader: tasksLoader,
            handle: {
              crumb: () => {
                return '手势模型体验';
              },
            },
            children: [
              {
                index: true,
                element: <GestureValidationPage />,
              },
            ],
          },
          {
            path: 'facesetting',
            element: <Outlet />,
            errorElement: <Page404 />,
            id: 'facesetting',
            loader: tasksLoader,
            handle: {
              crumb: () => {
                return '人脸训练参数设置';
              },
            },
            children: [
              {
                index: true,
                element: <FaceSettingPage />,
              },
            ],
          },
          {
            path: 'facecollection',
            element: <Outlet />,
            errorElement: <Page404 />,
            id: 'facecollection',
            loader: tasksLoader,
            handle: {
              crumb: () => {
                return '人脸视频采集';
              },
            },
            children: [
              {
                index: true,
                element: <FaceCollectionPage />,
              },
            ],
          },
          {
            path: 'facevalidation',
            element: <Outlet />,
            errorElement: <Page404 />,
            id: 'facevalidation',
            loader: tasksLoader,
            handle: {
              crumb: () => {
                return '人脸模型体验';
              },
            },
            children: [
              {
                index: true,
                element: <FaceValidationPage />,
              },
            ],
          },
          {
            path: 'UIUEConsole',
            element: <Outlet />,
            errorElement: <Page404 />,
            id: 'UIUEConsole',
            loader: tasksLoader,
            handle: {
              crumb: () => {
                return '用户控件功能';
              },
            },
            children: [
              {
                index: true,
                element: <UIUEConsolePage />,
              },
            ],
          },
          {
            path: 'UIUEUserControl',
            element: <Outlet />,
            errorElement: <Page404 />,
            id: 'UIUEUserControl',
            loader: tasksLoader,
            handle: {
              crumb: () => {
                return '用户操作日志';
              },
            },
            children: [
              {
                index: true,
                element: <UIUEUserControlPage />,
              },
            ],
          },
        ],
      },
    ],
  },
  {
    path: 'login',
    loader: loginLoader,
    element: <LoginPage />,
  },
  {
    path: 'register',
    loader: registerLoader,
    element: <Register />,
    handle: {
      crumb: () => {
        return '注册';
      },
    },
  },
  {
    path: '*',
    element: <Page404 />,
  },
];

export default routes;
